#pragma once

#include <pgg/profiling.h>
#include <pgg/database/database.h>
#include <pgg/database/performer/query_streamer.h>
#include <pgg/database/endpoint.h>
#include <pgg/error.h>
#include <pgg/no_exception.h>
#include <pgg/database/performer/error_handler_invoker.h>
#include <pgg/type_name.h>
#include <apq/error.hpp>
#include <boost/utility.hpp>
#include <boost/shared_ptr.hpp>

namespace pgg {
namespace database {

using Query = query::Query;

inline error_code errorCode( const apq::result& res, error_code::info_type info ) {
    auto msg = "in query " + info.query_name + " " + res.message();

    if (res.code() == apq::error::make_error_code(apq::error::unknown)) {
        return error_code(
                pgg::error_code (error::SqlErrors(res.sqlstate_code()), std::move(msg)),
                std::move(info));
    }

    return error_code(pgg::error_code(res.code(), std::move(msg)), std::move(info));
}

inline std::ostream & streamErrorInfo(std::ostream & s, const error_code::info_type & info) {
    s << "query:" << std::endl << info.query_text << ';' << std::endl;
    s << "values:" << std::endl;
    for (std::size_t i = 0; i < info.query_values.size(); i++) {
        s << "\t$" << i + 1 << '=' << info.query_values[i] << std::endl;
    }
    s << std::endl;
    s << "connInfo:" << std::endl << info.connstr;
    return s;
}

template <typename HandlerT>
class Context : public boost::enable_shared_from_this<Context<HandlerT>> {
    apq::query q;
    HandlerT h;
    profiling::LogPtr plog;
    logging::LogPtr log_;
    std::string queryName_;
    Time t;
    EndpointPtr endpoint_;
    LogError logQueryError_;

    template<typename... ArgsT>
    error_code invokeHandler(const apq::result& res, ArgsT&&... args) {
        const auto err = makeResult(res);
        if(err) {
            invokeErrorHandler(err);
        } else {
            invokeDataHandler(std::forward<ArgsT>(args)...);
        }
        return err;
    }

    template<typename... ArgsT>
    void invokeDataHandler(ArgsT&&... args) const {
        handler()(error_code(), std::forward<ArgsT>(args)...);
    }

    void invokeErrorHandler(error_code res) const {
        forwardError(handler(), res);
    }

    error_code::info_type makeErrorInfo() const {
        auto values = extractValues(query());
        return error_code::info_type{ connInfo(), queryName(), query().text_, std::move(values) };
    }

    error_code makeResult(const apq::result& res) const {
        if (res.code()) {
            return errorCode(res, makeErrorInfo());
        }
        return error_code();
    }

    Endpoint& endpoint() const { return *endpoint_; }

public:
    Context(const Query & query, HandlerT h, LoggingAttributes logAttrs,
            EndpointPtr endpoint)
    : q(query.text()), h(std::move(h)),
      plog(logAttrs.profiler), log_(logAttrs.logger), queryName_(query.name()),
      t(plog.get() ? now() : Time()),
      endpoint_(endpoint),
      logQueryError_(logAttrs.logQueryErrorPolicy) {
        query.mapValues(query::Mapper(q));
    }

    Context(const Context&) = delete;
    Context(Context&&) = delete;

    ~Context() noexcept {
        if (plog.get()) try {
            profiling::write(*plog, t, "QueryHandler ", queryName());
        } catch (...) {}
    }

    template<typename... ArgsT>
    void handle(ArgsT&&... args) noexcept {
        try {
            const auto error = invokeHandler(std::forward<ArgsT>(args)...);
            if (error) {
                logQueryError(error.message());
            }
        } catch (const std::exception& e) {
            logQueryError(e.what());
        } catch (...) {
            logQueryError("unknown exception");
        }
    }

    const apq::query & query() const { return q; }
    const std::string & queryName() const { return queryName_; }
    const std::string & connInfo() const { return endpoint().connectionInfo(); }
    HandlerT handler() const { return h; }
    const logging::LogPtr& log() const { return log_; }
    void logQueryError(const std::string & message) const {
        if (log() && (logQueryError_ == LogError::enable)) {
            using namespace logging;
            logError(*log(), [&] {
                std::ostringstream s;
                s << message << std::endl;
                streamErrorInfo(s, makeErrorInfo());

                Attributes attributes = {
                    QueryText(q.text_),
                    QueryValues(extractValues(q)),
                    ConnectionInfo(connInfo()),
                    OldMessage(s.str())
                };

                return Record{ queryName(), message, std::move(attributes) };
            });
        }
    }
};

} // namespace database
} // namespace pgg
