#pragma once

#include <boost/thread.hpp>
#include <boost/range/algorithm_ext/insert.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <map>

#include <pgg/database/endpoint.h>
#include <pgg/database/io_service.h>

namespace pgg {
namespace database {

class ConnectionPool {
public:
    ConnectionPool( IoServicePtr ioService, std::size_t limit, const ConnectionPoolSettings& settings )
        : limit(limit), ioService(ioService), settings(settings) {
    }

    typedef std::string ConnId;
    EndpointPtr get( const ConnId & id ) {
        Lock l(m);

        auto i = connections.find(id);
        while( i == connections.end() ) {
            if( inProgress(id) ) {
                i = wait(l, id);
            } else {
                return create(l, id);
            }
        }

        return i->second;
    }

    typedef std::map<ConnId, Endpoint::Stats> Stats;
    Stats getStats() const {
        Stats stats;
        Lock l(m);
        boost::insert(stats, connections | boost::adaptors::transformed(
            [] (const Endpoints::value_type & v) {
                return std::make_pair(v.first, v.second->getStats());
        }));
        l.unlock();

        return stats;
    }

private:
    typedef std::map<ConnId,EndpointPtr> Endpoints;
    typedef boost::mutex Mutex;
    typedef boost::unique_lock<boost::mutex> Lock;
    typedef std::set<ConnId> CreatingConnections;

    bool inProgress( const ConnId & id ) const {
        return creating.count(id);
    }

    Endpoints::iterator wait( Lock & l, const ConnId & id ) {
        cond.wait(l);
        return connections.find(id);
    }

    EndpointPtr create( Lock & l, const ConnId & id ) {
        processLimit();
        creating.insert(id);
        l.unlock();

        auto c = createEndpoint(ioService, id, settings);

        l.lock();
        creating.erase(id);
        connections.insert(std::make_pair(id, c));
        l.unlock();

        cond.notify_all();
        return c;
    }

    void processLimit() {
        if( limit && limit == (connections.size() + creating.size()) ) {
            throw std::runtime_error("macs::pg::ConnectionPool: Too much connections!");
        }
    }

    Endpoints connections;
    CreatingConnections creating;
    mutable Mutex m;
    boost::condition_variable cond;
    std::size_t limit;
    IoServicePtr ioService;
    ConnectionPoolSettings settings;
};

} // namespace database
} // namespace pgg
