#pragma once

#include <mail_errors/error_code.h>
#include <pgg/type_name.h>
#include <apq/error.hpp>
#include <apq/result.hpp>
#include <stdexcept>
#include <boost/asio.hpp>
#include <sstream>

namespace pgg {

using error_code = mail_errors::error_code;
using error_category = error_code::error_category;
using system_error = mail_errors::system_error;

namespace error {

/**
 * Common macs_pg error codes
 */
enum CommonErrors {
    ok, // No error
    noDataReceived, // No data receved by the query
    resolverTroubles, // Problem with uid/shard resolving
    endpointNumberOverflow, // Endpoint index is out of range
    noEndpointForRole, // No endpoints are received for the specified role
    exceptionInHandler, // Handler threw exception during invocation
    exceptionInConverter, // Result converter threw an exception
};

const error_category& getCommonCategory();

/**
 * These next categories relate to the PostgreSQL error codes, see "apq" project
 * error.hpp "apq::error::sqlstate::sqlstate_errors" for the actual error
 * code values
 */
const error_category& getConnectionCategory();
const error_category& getReadonlyCategory();
const error_category& getSqlCategory();

} // namespace error

namespace errc {

/**
 * Error conditions for database errors
 */
enum DatabaseErrors {
    userRemovedFromShard, // User not in this shard anymore, probably transfered to different shard
    databaseLockFailed,
    databaseReadOnly,
    communicationError,
    noEndpointFound,
};

const error_category& getDatabaseCategory();

} // namespace errc
} // namespace pgg

namespace boost {
namespace system {

template <>
struct is_error_code_enum<pgg::error::CommonErrors> {
    static const bool value = true;
};

template <>
struct is_error_condition_enum<pgg::errc::DatabaseErrors> {
    static const bool value = true;
};

} // namespace system
} // namespace boost

namespace pgg {
namespace error {

inline error_code::base_type make_error_code(CommonErrors e) {
    return error_code::base_type(static_cast<int>(e), getCommonCategory());
}

} // namespace error

namespace errc {

inline error_code::error_condition make_error_condition(DatabaseErrors e) {
    return error_code::error_condition(static_cast<int>(e), getDatabaseCategory());
}

} // namespace errc
} // namespace pgg

namespace boost {
namespace system {

template <>
struct is_error_code_enum<apq::error::sqlstate::sqlstate_errors> {
    static const bool value = true;
};

} // namespace system
} // namespace boost

namespace pgg {
namespace error {

using pgg::details::typeName;

using SqlErrors = apq::error::sqlstate::sqlstate_errors;

inline std::string apqSqlStateMessage(SqlErrors v) {
#define MACS_PG_APQ_SQL_STATE(value) case value: return #value;
    using namespace apq::error::sqlstate;
    switch (v) {
        MACS_PG_APQ_SQL_STATE(successful_completion)
        MACS_PG_APQ_SQL_STATE(warning)
        MACS_PG_APQ_SQL_STATE(dynamic_result_sets_returned)
        MACS_PG_APQ_SQL_STATE(implicit_zero_bit_padding)
        MACS_PG_APQ_SQL_STATE(null_value_eliminated_in_set_function)
        MACS_PG_APQ_SQL_STATE(privilege_not_granted)
        MACS_PG_APQ_SQL_STATE(privilege_not_revoked)
        MACS_PG_APQ_SQL_STATE(string_data_right_truncation_warning)
        MACS_PG_APQ_SQL_STATE(deprecated_feature)
        MACS_PG_APQ_SQL_STATE(no_data)
        MACS_PG_APQ_SQL_STATE(no_additional_dynamic_result_sets_returned)
        MACS_PG_APQ_SQL_STATE(sql_statement_not_yet_complete)
        MACS_PG_APQ_SQL_STATE(connection_exception)
        MACS_PG_APQ_SQL_STATE(connection_does_not_exist)
        MACS_PG_APQ_SQL_STATE(connection_failure)
        MACS_PG_APQ_SQL_STATE(sqlclient_unable_to_establish_sqlconnection)
        MACS_PG_APQ_SQL_STATE(sqlserver_rejected_establishment_of_sqlconnection)
        MACS_PG_APQ_SQL_STATE(transaction_resolution_unknown)
        MACS_PG_APQ_SQL_STATE(protocol_violation)
        MACS_PG_APQ_SQL_STATE(triggered_action_exception)
        MACS_PG_APQ_SQL_STATE(feature_not_supported)
        MACS_PG_APQ_SQL_STATE(invalid_transaction_initiation)
        MACS_PG_APQ_SQL_STATE(locator_exception)
        MACS_PG_APQ_SQL_STATE(invalid_locator_specification)
        MACS_PG_APQ_SQL_STATE(invalid_grantor)
        MACS_PG_APQ_SQL_STATE(invalid_grant_operation)
        MACS_PG_APQ_SQL_STATE(invalid_role_specification)
        MACS_PG_APQ_SQL_STATE(diagnostics_exception)
        MACS_PG_APQ_SQL_STATE(stacked_diagnostics_accessed_without_active_handler)
        MACS_PG_APQ_SQL_STATE(case_not_found)
        MACS_PG_APQ_SQL_STATE(cardinality_violation)
        MACS_PG_APQ_SQL_STATE(data_exception)
        MACS_PG_APQ_SQL_STATE(array_subscript_error)
        MACS_PG_APQ_SQL_STATE(character_not_in_repertoire)
        MACS_PG_APQ_SQL_STATE(datetime_field_overflow)
        MACS_PG_APQ_SQL_STATE(division_by_zero)
        MACS_PG_APQ_SQL_STATE(error_in_assignment)
        MACS_PG_APQ_SQL_STATE(escape_character_conflict)
        MACS_PG_APQ_SQL_STATE(indicator_overflow)
        MACS_PG_APQ_SQL_STATE(interval_field_overflow)
        MACS_PG_APQ_SQL_STATE(invalid_argument_for_logarithm)
        MACS_PG_APQ_SQL_STATE(invalid_argument_for_ntile_function)
        MACS_PG_APQ_SQL_STATE(invalid_argument_for_nth_value_function)
        MACS_PG_APQ_SQL_STATE(invalid_argument_for_power_function)
        MACS_PG_APQ_SQL_STATE(invalid_argument_for_width_bucket_function)
        MACS_PG_APQ_SQL_STATE(invalid_character_value_for_cast)
        MACS_PG_APQ_SQL_STATE(invalid_datetime_format)
        MACS_PG_APQ_SQL_STATE(invalid_escape_character)
        MACS_PG_APQ_SQL_STATE(invalid_escape_octet)
        MACS_PG_APQ_SQL_STATE(invalid_escape_sequence)
        MACS_PG_APQ_SQL_STATE(nonstandard_use_of_escape_character)
        MACS_PG_APQ_SQL_STATE(invalid_indicator_parameter_value)
        MACS_PG_APQ_SQL_STATE(invalid_parameter_value)
        MACS_PG_APQ_SQL_STATE(invalid_regular_expression)
        MACS_PG_APQ_SQL_STATE(invalid_row_count_in_limit_clause)
        MACS_PG_APQ_SQL_STATE(invalid_row_count_in_result_offset_clause)
        MACS_PG_APQ_SQL_STATE(invalid_time_zone_displacement_value)
        MACS_PG_APQ_SQL_STATE(invalid_use_of_escape_character)
        MACS_PG_APQ_SQL_STATE(most_specific_type_mismatch)
        MACS_PG_APQ_SQL_STATE(null_value_not_allowed)
        MACS_PG_APQ_SQL_STATE(null_value_no_indicator_parameter)
        MACS_PG_APQ_SQL_STATE(numeric_value_out_of_range)
        MACS_PG_APQ_SQL_STATE(string_data_length_mismatch)
        MACS_PG_APQ_SQL_STATE(string_data_right_truncation)
        MACS_PG_APQ_SQL_STATE(substring_error)
        MACS_PG_APQ_SQL_STATE(trim_error)
        MACS_PG_APQ_SQL_STATE(unterminated_c_string)
        MACS_PG_APQ_SQL_STATE(zero_length_character_string)
        MACS_PG_APQ_SQL_STATE(floating_point_exception)
        MACS_PG_APQ_SQL_STATE(invalid_text_representation)
        MACS_PG_APQ_SQL_STATE(invalid_binary_representation)
        MACS_PG_APQ_SQL_STATE(bad_copy_file_format)
        MACS_PG_APQ_SQL_STATE(untranslatable_character)
        MACS_PG_APQ_SQL_STATE(not_an_xml_document)
        MACS_PG_APQ_SQL_STATE(invalid_xml_document)
        MACS_PG_APQ_SQL_STATE(invalid_xml_content)
        MACS_PG_APQ_SQL_STATE(invalid_xml_comment)
        MACS_PG_APQ_SQL_STATE(invalid_xml_processing_instruction)
        MACS_PG_APQ_SQL_STATE(integrity_constraint_violation)
        MACS_PG_APQ_SQL_STATE(restrict_violation)
        MACS_PG_APQ_SQL_STATE(not_null_violation)
        MACS_PG_APQ_SQL_STATE(foreign_key_violation)
        MACS_PG_APQ_SQL_STATE(unique_violation)
        MACS_PG_APQ_SQL_STATE(check_violation)
        MACS_PG_APQ_SQL_STATE(exclusion_violation)
        MACS_PG_APQ_SQL_STATE(invalid_cursor_state)
        MACS_PG_APQ_SQL_STATE(invalid_transaction_state)
        MACS_PG_APQ_SQL_STATE(active_sql_transaction)
        MACS_PG_APQ_SQL_STATE(branch_transaction_already_active)
        MACS_PG_APQ_SQL_STATE(held_cursor_requires_same_isolation_level)
        MACS_PG_APQ_SQL_STATE(inappropriate_access_mode_for_branch_transaction)
        MACS_PG_APQ_SQL_STATE(inappropriate_isolation_level_for_branch_transaction)
        MACS_PG_APQ_SQL_STATE(no_active_sql_transaction_for_branch_transaction)
        MACS_PG_APQ_SQL_STATE(read_only_sql_transaction)
        MACS_PG_APQ_SQL_STATE(schema_and_data_statement_mixing_not_supported)
        MACS_PG_APQ_SQL_STATE(no_active_sql_transaction)
        MACS_PG_APQ_SQL_STATE(in_failed_sql_transaction)
        MACS_PG_APQ_SQL_STATE(invalid_sql_statement_name)
        MACS_PG_APQ_SQL_STATE(triggered_data_change_violation)
        MACS_PG_APQ_SQL_STATE(invalid_authorization_specification)
        MACS_PG_APQ_SQL_STATE(invalid_password)
        MACS_PG_APQ_SQL_STATE(dependent_privilege_descriptors_still_exist)
        MACS_PG_APQ_SQL_STATE(dependent_objects_still_exist)
        MACS_PG_APQ_SQL_STATE(invalid_transaction_termination)
        MACS_PG_APQ_SQL_STATE(sql_routine_exception)
        MACS_PG_APQ_SQL_STATE(function_executed_no_return_statement)
        MACS_PG_APQ_SQL_STATE(modifying_sql_data_not_permitted)
        MACS_PG_APQ_SQL_STATE(prohibited_sql_statement_attempted)
        MACS_PG_APQ_SQL_STATE(reading_sql_data_not_permitted)
        MACS_PG_APQ_SQL_STATE(invalid_cursor_name)
        MACS_PG_APQ_SQL_STATE(external_routine_exception)
        MACS_PG_APQ_SQL_STATE(containing_sql_not_permitted)
        MACS_PG_APQ_SQL_STATE(modifying_sql_data_not_permitted_external)
        MACS_PG_APQ_SQL_STATE(prohibited_sql_statement_attempted_external)
        MACS_PG_APQ_SQL_STATE(reading_sql_data_not_permitted_external)
        MACS_PG_APQ_SQL_STATE(external_routine_invocation_exception)
        MACS_PG_APQ_SQL_STATE(invalid_sqlstate_returned)
        MACS_PG_APQ_SQL_STATE(null_value_not_allowed_external)
        MACS_PG_APQ_SQL_STATE(trigger_protocol_violated)
        MACS_PG_APQ_SQL_STATE(srf_protocol_violated)
        MACS_PG_APQ_SQL_STATE(savepoint_exception)
        MACS_PG_APQ_SQL_STATE(invalid_savepoint_specification)
        MACS_PG_APQ_SQL_STATE(invalid_catalog_name)
        MACS_PG_APQ_SQL_STATE(invalid_schema_name)
        MACS_PG_APQ_SQL_STATE(transaction_rollback)
        MACS_PG_APQ_SQL_STATE(transaction_integrity_constraint_violation)
        MACS_PG_APQ_SQL_STATE(serialization_failure)
        MACS_PG_APQ_SQL_STATE(statement_completion_unknown)
        MACS_PG_APQ_SQL_STATE(deadlock_detected)
        MACS_PG_APQ_SQL_STATE(syntax_error_or_access_rule_violation)
        MACS_PG_APQ_SQL_STATE(syntax_error)
        MACS_PG_APQ_SQL_STATE(insufficient_privilege)
        MACS_PG_APQ_SQL_STATE(cannot_coerce)
        MACS_PG_APQ_SQL_STATE(grouping_error)
        MACS_PG_APQ_SQL_STATE(windowing_error)
        MACS_PG_APQ_SQL_STATE(invalid_recursion)
        MACS_PG_APQ_SQL_STATE(invalid_foreign_key)
        MACS_PG_APQ_SQL_STATE(invalid_name)
        MACS_PG_APQ_SQL_STATE(name_too_long)
        MACS_PG_APQ_SQL_STATE(reserved_name)
        MACS_PG_APQ_SQL_STATE(datatype_mismatch)
        MACS_PG_APQ_SQL_STATE(indeterminate_datatype)
        MACS_PG_APQ_SQL_STATE(collation_mismatch)
        MACS_PG_APQ_SQL_STATE(indeterminate_collation)
        MACS_PG_APQ_SQL_STATE(wrong_object_type)
        MACS_PG_APQ_SQL_STATE(undefined_column)
        MACS_PG_APQ_SQL_STATE(undefined_function)
        MACS_PG_APQ_SQL_STATE(undefined_table)
        MACS_PG_APQ_SQL_STATE(undefined_parameter)
        MACS_PG_APQ_SQL_STATE(undefined_object)
        MACS_PG_APQ_SQL_STATE(duplicate_column)
        MACS_PG_APQ_SQL_STATE(duplicate_cursor)
        MACS_PG_APQ_SQL_STATE(duplicate_database)
        MACS_PG_APQ_SQL_STATE(duplicate_function)
        MACS_PG_APQ_SQL_STATE(duplicate_prepared_statement)
        MACS_PG_APQ_SQL_STATE(duplicate_schema)
        MACS_PG_APQ_SQL_STATE(duplicate_table)
        MACS_PG_APQ_SQL_STATE(duplicate_alias)
        MACS_PG_APQ_SQL_STATE(duplicate_object)
        MACS_PG_APQ_SQL_STATE(ambiguous_column)
        MACS_PG_APQ_SQL_STATE(ambiguous_function)
        MACS_PG_APQ_SQL_STATE(ambiguous_parameter)
        MACS_PG_APQ_SQL_STATE(ambiguous_alias)
        MACS_PG_APQ_SQL_STATE(invalid_column_reference)
        MACS_PG_APQ_SQL_STATE(invalid_column_definition)
        MACS_PG_APQ_SQL_STATE(invalid_cursor_definition)
        MACS_PG_APQ_SQL_STATE(invalid_database_definition)
        MACS_PG_APQ_SQL_STATE(invalid_function_definition)
        MACS_PG_APQ_SQL_STATE(invalid_prepared_statement_definition)
        MACS_PG_APQ_SQL_STATE(invalid_schema_definition)
        MACS_PG_APQ_SQL_STATE(invalid_table_definition)
        MACS_PG_APQ_SQL_STATE(invalid_object_definition)
        MACS_PG_APQ_SQL_STATE(with_check_option_violation)
        MACS_PG_APQ_SQL_STATE(insufficient_resources)
        MACS_PG_APQ_SQL_STATE(disk_full)
        MACS_PG_APQ_SQL_STATE(out_of_memory)
        MACS_PG_APQ_SQL_STATE(too_many_connections)
        MACS_PG_APQ_SQL_STATE(configuration_limit_exceeded)
        MACS_PG_APQ_SQL_STATE(program_limit_exceeded)
        MACS_PG_APQ_SQL_STATE(statement_too_complex)
        MACS_PG_APQ_SQL_STATE(too_many_columns)
        MACS_PG_APQ_SQL_STATE(too_many_arguments)
        MACS_PG_APQ_SQL_STATE(object_not_in_prerequisite_state)
        MACS_PG_APQ_SQL_STATE(object_in_use)
        MACS_PG_APQ_SQL_STATE(cant_change_runtime_param)
        MACS_PG_APQ_SQL_STATE(lock_not_available)
        MACS_PG_APQ_SQL_STATE(operator_intervention)
        MACS_PG_APQ_SQL_STATE(query_canceled)
        MACS_PG_APQ_SQL_STATE(admin_shutdown)
        MACS_PG_APQ_SQL_STATE(crash_shutdown)
        MACS_PG_APQ_SQL_STATE(cannot_connect_now)
        MACS_PG_APQ_SQL_STATE(database_dropped)
        case sqlstate_errors::system_error: return "system_error";
        MACS_PG_APQ_SQL_STATE(io_error)
        MACS_PG_APQ_SQL_STATE(undefined_file)
        MACS_PG_APQ_SQL_STATE(duplicate_file)
        MACS_PG_APQ_SQL_STATE(config_file_error)
        MACS_PG_APQ_SQL_STATE(lock_file_exists)
        MACS_PG_APQ_SQL_STATE(fdw_error)
        MACS_PG_APQ_SQL_STATE(fdw_column_name_not_found)
        MACS_PG_APQ_SQL_STATE(fdw_dynamic_parameter_value_needed)
        MACS_PG_APQ_SQL_STATE(fdw_function_sequence_error)
        MACS_PG_APQ_SQL_STATE(fdw_inconsistent_descriptor_information)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_attribute_value)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_column_name)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_column_number)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_data_type)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_data_type_descriptors)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_descriptor_field_identifier)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_handle)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_option_index)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_option_name)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_string_length_or_buffer_length)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_string_format)
        MACS_PG_APQ_SQL_STATE(fdw_invalid_use_of_null_pointer)
        MACS_PG_APQ_SQL_STATE(fdw_too_many_handles)
        MACS_PG_APQ_SQL_STATE(fdw_out_of_memory)
        MACS_PG_APQ_SQL_STATE(fdw_no_schemas)
        MACS_PG_APQ_SQL_STATE(fdw_option_name_not_found)
        MACS_PG_APQ_SQL_STATE(fdw_reply_handle)
        MACS_PG_APQ_SQL_STATE(fdw_schema_not_found)
        MACS_PG_APQ_SQL_STATE(fdw_table_not_found)
        MACS_PG_APQ_SQL_STATE(fdw_unable_to_create_execution)
        MACS_PG_APQ_SQL_STATE(fdw_unable_to_create_reply)
        MACS_PG_APQ_SQL_STATE(fdw_unable_to_establish_connection)
        MACS_PG_APQ_SQL_STATE(plpgsql_error)
        MACS_PG_APQ_SQL_STATE(raise_exception)
        MACS_PG_APQ_SQL_STATE(no_data_found)
        MACS_PG_APQ_SQL_STATE(too_many_rows)
        MACS_PG_APQ_SQL_STATE(internal_error)
        MACS_PG_APQ_SQL_STATE(data_corrupted)
        MACS_PG_APQ_SQL_STATE(index_corrupted)
        MACS_PG_APQ_SQL_STATE(custom_error_mdba1)
    }
    return "unknown error";
#undef MACS_PG_APQ_SQL_STATE
}

class CommonCategory : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "pgg::error::CommonCategory";
    }

    std::string message(int v) const override {
        switch(v) {
            case ok :
                return "no error";
            case noDataReceived :
                return "no result data received from query";
            case resolverTroubles:
                return "some troubles were appeared during resolving the uid/shard";
            case endpointNumberOverflow:
                return "endpoint number is greater than maximum available";
            case noEndpointForRole:
                return "no endpoints are received for the specified role";
            case exceptionInHandler:
                return "handler threw exception during invocation";
            case exceptionInConverter:
                return "result converter threw exception";
            default:
                return "unknown error";
        }
    }
};

class ReadonlyCategory : public boost::system::error_category {
public:
    static bool belongs(int v) {
        return v == apq::error::sqlstate::read_only_sql_transaction;
    }

    const char* name() const noexcept override {
        return "pgg::error::ReadonlyCategory";
    }

    std::string message(int v) const override {
        return apqSqlStateMessage(SqlErrors(v));
    }

    bool equivalent(const boost::system::error_code& code, int condition) const noexcept override final {
        return boost::system::error_category::equivalent(code, condition);
    }

    bool equivalent(int code, const boost::system::error_condition& condition) const noexcept override final {
        return boost::system::error_category::equivalent(code, condition);
    }
};

class ConnectionCategory : public boost::system::error_category {
public:
    static bool belongs(int v) {
        using namespace apq::error::sqlstate;
        switch(v) {
            case connection_exception:
            case connection_does_not_exist:
            case connection_failure:
            case sqlclient_unable_to_establish_sqlconnection:
            case sqlserver_rejected_establishment_of_sqlconnection:
            case transaction_resolution_unknown:
            case protocol_violation:
                return true;
            default:
                return false;
        }
    }

    const char* name() const noexcept override {
        return "pgg::error::ConnectionCategory";
    }

    std::string message(int v) const override {
        return apqSqlStateMessage(SqlErrors(v));
    }
};

class SqlCategory : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "pgg::error::SqlCategory";
    }

    std::string message(int v) const override {
        return apqSqlStateMessage(SqlErrors(v));
    }

    bool equivalent(const boost::system::error_code& code, int condition) const noexcept override final {
        return boost::system::error_category::equivalent(code, condition);
    }

    bool equivalent(int code, const boost::system::error_condition& condition) const noexcept override final {
        return boost::system::error_category::equivalent(code, condition);
    }

    static bool goodSqlStates (int v) noexcept {
        using namespace apq::error::sqlstate;
        switch (v) {
            case successful_completion:
            case warning:
            case dynamic_result_sets_returned:
            case implicit_zero_bit_padding:
            case null_value_eliminated_in_set_function:
            case privilege_not_granted:
            case privilege_not_revoked:
            case string_data_right_truncation_warning:
            case deprecated_feature:
                return true;
            default:
                return false;
        }
    }
};

inline error_code::base_type make_error_code(SqlErrors err) {
    using T = error_code::base_type;
    const int e = static_cast<int>(err);
    if (SqlCategory::goodSqlStates(e)) {
        return T();
    } else if (ConnectionCategory::belongs(e)) {
        return T(e, getConnectionCategory());
    } else if (ReadonlyCategory::belongs(e)) {
        return T(e, getReadonlyCategory());
    }
    return T(e, getSqlCategory());
}

inline bool isCommunicationError(const error_code& code) {
    if (code.category() == boost::asio::error::get_system_category()) {
        using namespace boost::asio::error;
        switch (code.default_error_condition().value()) {
            case broken_pipe:
            case connection_aborted:
            case connection_refused:
            case connection_reset:
            case fault:
            case host_unreachable:
            case interrupted:
            case network_reset:
            case not_connected:
            case operation_aborted:
            case shut_down:
            case timed_out:
            case try_again:
                return true;
            default:
                return false;
        }
    } else {
        return code.category() == error::getConnectionCategory() ||
               code.category() == boost::asio::error::get_netdb_category() ||
               code == apq::error::request_queue_timed_out ||
               code == apq::error::network ||
               code == boost::system::errc::io_error;
    }
}

} // namespace error

namespace errc {

class DatabaseCategory: public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "pgg::errc::DatabaseCategory";
    }

    std::string message(int v) const override {
        switch (v) {
            case userRemovedFromShard:
                return "user removed from this shard, probably transfered to different shard";

            case databaseLockFailed:
                return "database lock failed";

            case databaseReadOnly:
                return "database in read-only state";

            case communicationError:
                return "communication error";
            case noEndpointFound:
                return "no endpoint found";
            default:
                return "unknown condition";
        }
    }

    bool equivalent(const boost::system::error_code& code, int condition) const noexcept override final {
        switch (condition) {
        case pgg::errc::userRemovedFromShard:
            return code == apq::error::sqlstate::custom_error_mdba1;
        case pgg::errc::databaseLockFailed:
            return code == apq::error::sqlstate::lock_not_available;
        case pgg::errc::databaseReadOnly:
            return code == apq::error::sqlstate::read_only_sql_transaction;
        case pgg::errc::communicationError:
            return pgg::error::isCommunicationError(code);
        case pgg::errc::noEndpointFound:
            return code == error::noEndpointForRole || code == error::endpointNumberOverflow;
        };
        return boost::system::error_category::equivalent(code, condition);
    }

    bool equivalent(int code, const boost::system::error_condition& condition) const noexcept override final {
        return boost::system::error_category::equivalent(code, condition);
    }
};

} // namespace errc
} // namespace pgg

namespace apq {
namespace error {
namespace sqlstate {

inline auto make_error_code(sqlstate_errors err) ->
        decltype(::pgg::error::make_error_code(err)) {
    return ::pgg::error::make_error_code(err);
}

} // namespace sqlstate
} // namespace error
} // namespace apq
