#pragma once

#include <pgg/database/pool.h>
#include <pgg/database/endpoint_provider.h>
#include <pgg/database/fallback.h>
#include <pgg/query/repository.h>
#include <pgg/query/conf.h>
#include <pgg/query/transaction.h>
#include <pgg/query/query_register.h>
#include <pgg/request_executor.h>
#include <pgg/service/uid_resolver.h>

namespace pgg {

using QueryHandleStrategy = database::fallback::StrategyProvider;
using QueryConf = boost::shared_ptr<query::Repository>;

template <typename... Qs, typename... Ps>
QueryConf readQueryConfFile(const std::string& fileName, query::RegisterQueries<Qs...>,
                            query::RegisterParameters<Ps...> paramsReg) {
    using namespace query;
    Configuration cfg;
    cfg.loadFromFile(fileName);
    return boost::make_shared<Repository>(cfg.queries(), cfg.parameters(),
        RegisterQueries<Begin, Commit, Rollback, Qs...>(), paramsReg);
}

using ReadMasterThenReplica = database::fallback::strategy::ReadMasterThenReplica;
using MasterOnly = database::fallback::strategy::MasterOnly;
using ReadReplicaThenMaster = database::fallback::strategy::ReadReplicaThenMaster;
using ReplicaOnly = database::fallback::strategy::ReplicaOnly;
using ReadNoLagReplicaThenMasterThenReplica = database::fallback::strategy::ReadNoLagReplicaThenMasterThenReplica;

struct FactoryParams {
    ConnectionPoolPtr connPool;
    profiling::LogPtr profiler;
    query::RepositoryPtr queryConf;
    logging::LogPtr logger;
    Credentials credentials;
    RequestInfo requestInfo;
};

class Factory {
public:
    using Self = Factory;
    using Uid = UidResolver::Id;
    using ShardId = ShardResolver::Id;

    template <typename Rules = ReadMasterThenReplica, typename Id, typename ResolverFactory>
    static RequestExecutor product(const Id& id, const ResolverFactory& resolverFactory,
                                   FactoryParams params, Rules autoQueryHandleRules = Rules()) {
        if (!params.queryConf) {
            throw std::invalid_argument("no queryConf specified!");
        }

        QueryHandleStrategy autoQueryHandleStartegy = autoQueryHandleRules;
        auto resolver = resolverFactory->product(std::move(params.credentials), std::move(params.requestInfo));
        auto generator = createDatabaseGenerator(
            id, std::move(resolver), std::move(params.connPool), std::move(params.profiler),
            std::move(params.logger), std::move(autoQueryHandleStartegy));
        return RequestExecutor(std::move(params.queryConf), std::move(generator));
    }

private:
    template <typename Resolver>
    static DatabasePtr createDatabase(const database::Traits<Resolver>& traits,
                                      database::LoggingAttributes logAttrs,
                                      database::EndpointQuery endpoint) {
        using namespace database;
        using Impl = DatabaseImpl<ResolverProvider<Resolver>>;
        return boost::make_shared<Impl>(ResolverProvider<Resolver>(traits, endpoint), std::move(logAttrs));
    }

    template <typename Id, typename Resolver>
    static fb::DatabaseGenerator createDatabaseGenerator(
                        const Id& id, std::shared_ptr<Resolver> resolver,
                        ConnectionPoolPtr connPool, profiling::LogPtr profiler,
                        logging::LogPtr logger, QueryHandleStrategy autoQueryHandleStartegy) {
        using ResolverType = std::remove_const_t<Resolver>;

        auto dbf = [
            traits = database::Traits<ResolverType>(id, std::move(connPool), std::move(resolver)),
            profiler = std::move(profiler),
            logger = logger
        ]( database::EndpointQuery endpoint, LogError logError ){
            return createDatabase(traits, {std::move(profiler), std::move(logger), logError},
                                  std::move(endpoint));
        };

        return fb::DatabaseGenerator(std::move(dbf), std::move(autoQueryHandleStartegy),
                                     std::move(logger));
    }
};

} // namespace pgg
