#pragma once

#include <pgg/error.h>
#include <pgg/cast.h>
#include <apq/row_iterator.hpp>
#include <io_result/io_result.h>
#include <pgg/signature.h>
#include <boost/optional.hpp>

namespace pgg {
namespace hooks {

template <typename T>
struct RowImpl {
    static_assert(signature::detail::false_t<T>::value, "Cursor type is not supported");
    static decltype(auto) apply(const T&);
};

template <typename T>
struct ContinuationImpl {
    static_assert(signature::detail::false_t<T>::value, "Cursor type is not supported");
    static decltype(auto) apply(const T&);
};

template <typename T>
struct GetContextImpl {
    static decltype(auto) apply(T& cursor) {
        return cursor.get();
    }
};

namespace detail {

template <typename Reflection, typename Getter, typename Row>
auto result(Getter&& get, Row&& row, error_code& ec)
        -> boost::optional<decltype(get(cast<Reflection>(row)))> {
    try {
        return get(cast<Reflection>(row));
    } catch (const boost::system::system_error& e) {
        ec = error_code{e.code(), e.what()};
    } catch (const std::exception& e) {
        ec = error_code{error::exceptionInConverter, e.what()};
    }
    return {};
}

} // namespace detail

template <typename Cursor>
decltype(auto) row(Cursor&& cursor) {
    return RowImpl<std::decay_t<Cursor>>::apply(std::forward<Cursor>(cursor));
}

template <typename Cursor>
decltype(auto) continuation(Cursor&& cursor) {
    return ContinuationImpl<std::decay_t<Cursor>>::apply(std::forward<Cursor>(cursor));
}

template <typename Cursor>
decltype(auto) getContext(Cursor& cursor) {
    return GetContextImpl<Cursor>::apply(cursor);
}

template <class Reflection, class Hook, class GetValue>
struct SequenceWrapper {
    struct WrappedHook {
        Hook hook;
        GetValue get;

        template <typename Cursor>
        void operator()(error_code err, Cursor c = Cursor()) const {
            if (err) {
                hook(std::move(err));
            } else if (auto ctx = getContext(c)) {
                auto r = std::move(row(*ctx));
                error_code ec;
                auto res = detail::result<Reflection>(get, *r, ec);
                if (ec) {
                    hook(std::move(ec));
                } else {
                    hook({std::move(*res), [ctx]{ continuation(*ctx).resume();}});
                }
            } else {
                hook();
            }
        }
    };

    using type = SequenceWrapper;
    static auto wrap(Hook hook, GetValue get) {
        return WrappedHook{ std::move(hook), std::move(get) };
    }
};

template <class Reflection, class Hook, class GetValue>
struct OptionalWrapper {
    using type = OptionalWrapper;
    static auto wrap(Hook hook, GetValue get) {
        return [hook = std::move(hook), get = std::move(get)]
                (error_code err, auto data) {
            if constexpr (std::is_same_v<decltype(data), boost::none_t>) {
                hook(std::move(err));
            } else {
                if (err) {
                    hook(std::move(err));
                } else if (data.empty()) {
                    hook();
                } else {
                    error_code ec;
                    auto res = detail::result<Reflection>(get, data.front(), ec);
                    if (ec) {
                        hook(std::move(ec));
                    } else {
                        hook(std::move(*res));
                    }
                }
            }
        };
    }
};

template <typename Reflection, typename Hook, typename GetValue>
struct OnceSequenceWrapper {
    using type = OnceSequenceWrapper;
    template <typename Convert>
    static auto wrap(Hook hook, GetValue get, Convert conv) {
        return [hook = std::move(hook), get = std::move(get), conv = std::move(conv)]
                (error_code err, auto data) {
            if constexpr (std::is_same_v<decltype(data), boost::none_t>) {
                hook(std::move(err));
            } else {
                if (err) {
                    hook(std::move(err));
                } else {
                    std::vector<std::invoke_result_t<GetValue, Reflection>> items;
                    for (const auto& row : data) {
                        error_code ec;
                        auto res = detail::result<Reflection>(get, row, ec);
                        if (ec) {
                            return hook(std::move(ec));
                        } else {
                            items.push_back(*res);
                        }
                    }
                    hook(conv(std::move(items)));
                }
            }
        };
    }

    static auto wrap(Hook hook, GetValue get) {
        return [hook = std::move(hook), get = std::move(get)]
                (error_code err, auto data) {
            if constexpr (std::is_same_v<decltype(data), boost::none_t>) {
                hook(std::move(err));
            } else {
                if (err) {
                    hook(std::move(err));
                } else {
                    typename Hook::Arg items;
                    for (const auto& row : data) {
                        error_code ec;
                        auto res = detail::result<Reflection>(get, row, ec);
                        if (ec) {
                            return hook(std::move(ec));
                        } else {
                            items.push_back(*res);
                        }
                    }
                    hook(std::move(items));
                }
            }
        };
    }
};

template <typename Reflection, typename Hook, typename GetValue>
struct OnceWrapper {
    using type = OnceWrapper;
    template <typename Details>
    static auto wrap(Hook hook, GetValue get, Details details) {
        return [hook = std::move(hook), get = std::move(get), details = std::move(details)]
                (error_code err, auto data) {
            if constexpr (std::is_same_v<decltype(data), boost::none_t>) {
                hook(std::move(err));
            } else {
                if (err) {
                    hook(std::move(err));
                } else if (data.empty()) {
                    std::ostringstream s;
                    s << "empty result from database, details: ";
                    details(s);
                    hook(error_code(error::noDataReceived, s.str()));
                } else {
                    error_code ec;
                    auto res = detail::result<Reflection>(get, data.front(), ec);
                    if (ec) {
                        hook(std::move(ec));
                    } else {
                        hook(std::move(*res));
                    }
                }
            }
        };
    }

    static auto wrap(Hook hook, GetValue get) {
        return wrap(std::move(hook), std::move(get), [](auto& s) { s << "no details";});
    }
};

template <typename Reflection, typename Hook, typename GetValue>
struct SelectWrapper {

    template <typename CondT, typename ThenT, typename ElseT>
    using If = boost::mpl::eval_if< CondT, ThenT, ElseT >;
    template <typename CondT>
    using Else = CondT;
    template <typename CondT, typename ThenT, typename ElseT>
    using Elif = Else<If< CondT, ThenT, ElseT >>;

    template <typename T>
    using Decay = typename boost::remove_const<T>::type;

    using StartegyType = io_result::hooks::Strategy::type<Hook>;

    template <typename T>
    using StartegyTypeIs = boost::is_same<StartegyType, T>;

    using GetValueRes = std::invoke_result_t<GetValue, Reflection>;

    using Arg = typename Hook::Arg;

    using Selector =
    If< StartegyTypeIs<io_result::hooks::Strategy::Sequence>,
        SequenceWrapper<Reflection, Hook, GetValue>,
    Elif< StartegyTypeIs<io_result::hooks::Strategy::Optional>,
        OptionalWrapper<Reflection, Hook, GetValue>,
    Elif< boost::is_same<Arg, GetValueRes>, //StartegyTypeIs<io_result::hooks::Strategy::Once>,
        OnceWrapper<Reflection, Hook, GetValue>,
    Else<
        OnceSequenceWrapper<Reflection, Hook, GetValue>
    >>>>;

    using type = typename Selector::type;
};


} // namespace hooks


template <class Reflection, class Hook, class GetValue, typename ... Args>
auto wrapHook(Hook hook, GetValue get, Args&& ... args) {
    using Wrapper = typename hooks::SelectWrapper<Reflection, Hook, GetValue>::type;
    return Wrapper::wrap(std::move(hook), std::move(get), std::forward<Args>(args)...);
}

template <typename Reflection, typename Handler, typename GetValue, typename... Args>
auto wrapHandler(Handler handler, GetValue get, Args&&... args) {
    using Hook = signature::select_hook_type<Handler>;
    using Wrapper = typename hooks::SelectWrapper<Reflection, Hook, GetValue>::type;
    return Wrapper::wrap(Hook(std::move(handler)), std::move(get), std::forward<Args>(args)...);
}

} // namespace pgg
