#pragma once

#include <pgg/logging.h>
#include <logdog/logger.h>

namespace pgg {
namespace logging {

LOGDOG_DEFINE_ATTRIBUTE(std::string, macs_message)
LOGDOG_DEFINE_ATTRIBUTE(std::string, macs_method)
LOGDOG_DEFINE_ATTRIBUTE(std::string, query_text)
LOGDOG_DEFINE_ATTRIBUTE(std::vector<std::string>, query_values)
LOGDOG_DEFINE_ATTRIBUTE(std::string, connection_info)
LOGDOG_DEFINE_ATTRIBUTE(std::string, rule_name)
LOGDOG_DEFINE_ATTRIBUTE(std::string_view, strategy_name)

using ConnectionInfoView = std::function<ConnectionInfo(const ConnectionInfo& c)>;

template <typename Logger>
class TypedLog : public Log {
private:
    static auto makeLogAttrs(const Method& method,
                             const Message& message,
                             const ConnectionInfoView& connInfo,
                             Attributes attributes) {

        auto attrs = std::make_tuple(
            macs_message=message,
            macs_method=method,
            query_text=logdog::none,
            query_values=logdog::none,
            connection_info=logdog::none,
            rule_name=logdog::none,
            strategy_name=logdog::none,
            logdog::exception=logdog::none,
            logdog::error_code=logdog::none
        );
        using OptionalAttrs = decltype(attrs);

        struct visitor : public boost::static_visitor<void> {
            void operator()(const QueryText& v) const {
                logdog::update_attribute(attrs, query_text, v);
            }
            void operator()(const QueryValues& v) const {
                logdog::update_attribute(attrs, query_values, v);
            }
            void operator()(const ConnectionInfo& v) const {
                logdog::update_attribute(attrs, connection_info, connInfo(v));
            }
            void operator()(const OldMessage&) const {}
            void operator()(const RuleName& v) const {
                logdog::update_attribute(attrs, rule_name, v);
            }
            void operator()(const StrategyName& v) const {
                logdog::update_attribute(attrs, strategy_name, v);
            }
            void operator()(const error_code* v) const {
                logdog::update_attribute(attrs, logdog::error_code, *v);
            }
            void operator()(const std::exception* v) const {
                logdog::update_attribute(attrs, logdog::exception, *v);
            }

            visitor(OptionalAttrs& attrs, const ConnectionInfoView& connInfo)
                : attrs(attrs)
                , connInfo(connInfo)
            { }

            OptionalAttrs& attrs;
            const ConnectionInfoView& connInfo;
        };

        for(const auto& attr: attributes) {
            boost::apply_visitor(visitor(attrs, connInfo), attr);
        }

        return attrs;
    }

public:
    TypedLog(Logger logger)
        : logger(std::move(logger))
        , connInfoView([](const auto& c) { return c; })
    { }

    TypedLog(Logger logger, ConnectionInfoView view)
        : logger(std::move(logger))
        , connInfoView(view)
    { }

    using LogAttrs = std::invoke_result_t<decltype(&makeLogAttrs), const Method&, const Message&, const ConnectionInfoView&, Attributes>;
    void logTyped(Level level, const LogAttrs& attrs) {
        switch (level) {
        case Level::Warning:
            logger.write(logdog::warning, attrs);
            break;
        case Level::Error:
            logger.write(logdog::error, attrs);
            break;
        case Level::Notice:
            logger.write(logdog::notice, attrs);
            break;
        case Level::Debug:
            logger.write(logdog::debug, attrs);
        }
    }

    void log(const Level level, const Method &method,
             const Message &message, Attributes attributes) final override {
        logTyped(level, makeLogAttrs(method, message, connInfoView, std::move(attributes)));
    }

    bool applicable(const Level level) final override {
        switch (level) {
        case Level::Warning:
            return logger.applicable(logdog::warning);
        case Level::Error:
            return logger.applicable(logdog::error);
        case Level::Notice:
            return logger.applicable(logdog::notice);
        case Level::Debug:
            return logger.applicable(logdog::debug);
        }
        return false;
    }

protected:
    Logger logger;
    ConnectionInfoView connInfoView;
};

template <typename Logger>
auto makeTypedLog(Logger logger) {
    return std::make_shared<TypedLog<Logger>>(logger);
}

} // namespace logging
} // namespace pgg

