#pragma once

#include <pgg/error.h>
#include <boost/shared_ptr.hpp>
#include <string>
#include <vector>
#include <boost/variant.hpp>
#include <boost/serialization/strong_typedef.hpp>

namespace pgg {
namespace logging {

using Message = std::string;
using Method = std::string;

BOOST_STRONG_TYPEDEF(std::string, QueryText)
BOOST_STRONG_TYPEDEF(std::vector<std::string>, QueryValues)
BOOST_STRONG_TYPEDEF(std::string, ConnectionInfo)
BOOST_STRONG_TYPEDEF(std::string, RuleName)
BOOST_STRONG_TYPEDEF(std::string_view, StrategyName)
BOOST_STRONG_TYPEDEF(Message, OldMessage)

using Attribute = boost::variant<QueryText, QueryValues, ConnectionInfo, OldMessage,
                                 RuleName, StrategyName, const pgg::error_code*, const std::exception*>;
using Attributes = std::vector<Attribute>;

enum class Level {
    Warning,
    Error,
    Notice,
    Debug
};

class Log {
public:
    virtual ~Log() = default;

    virtual void log(const Level level, const Method& method, const Message& message,
        Attributes attributes) = 0;

    virtual bool applicable(const Level level) = 0;
};

using LogPtr = std::shared_ptr<Log>;

struct Record {
    Method method;
    Message message;
    Attributes attributes;
};

template <typename RecordProducer>
void logWithLevel(Log& log, const Level level, RecordProducer&& producer) {
    if (log.applicable(level)) {
        auto record = producer();
        log.log(level, record.method, record.message, std::move(record.attributes));
    }
}

template <typename RecordProducer>
void logError(Log& log, RecordProducer&& producer) {
    logWithLevel(log, Level::Error, std::forward<RecordProducer>(producer));
}

template <typename RecordProducer>
void logWarning(Log& log, RecordProducer&& producer) {
    logWithLevel(log, Level::Warning, std::forward<RecordProducer>(producer));
}

template <typename RecordProducer>
void logNotice(Log& log, RecordProducer&& producer) {
    logWithLevel(log, Level::Notice, std::forward<RecordProducer>(producer));
}

template <typename RecordProducer>
void logDebug(Log& log, RecordProducer&& producer) {
    logWithLevel(log, Level::Debug, std::forward<RecordProducer>(producer));
}

} // namespace logging
} // namespace pgg
