#pragma once

#include <boost/bind.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/make_shared.hpp>
#include <boost/enable_shared_from_this.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/iterator/indirect_iterator.hpp>
#include <boost/range.hpp>
#include <boost/range/adaptors.hpp>
#include <boost/range/algorithm.hpp>
#include <vector>
#include <string>
#include <algorithm>
#include <set>

namespace pgg {
namespace query {

typedef std::map<std::string, std::size_t> VariablesMap;
typedef std::set<std::string> VariablesSet;

class BodyPart {
public:
    typedef boost::shared_ptr<BodyPart const> Ptr;
    typedef std::vector<std::string> Variables;
    virtual ~BodyPart() {};
    virtual void stream(std::string & s) const = 0;
    virtual const std::string & text() const = 0;
    virtual Ptr resolve( const VariablesMap & map, VariablesSet & used ) const = 0;
    virtual bool resolved() const = 0;
    virtual bool textPart() const = 0;
    virtual Ptr assign( const Variables & v ) const = 0;
};

class Body {
public:
    typedef BodyPart Part;
    class Text : public Part, public boost::enable_shared_from_this<Text> {
        std::string text_;
    public:
        Text( const std::string & v ) : text_(v) {}

        void stream(std::string & s) const override { s.append(text()); }

        const std::string & text() const override { return text_; }

        Ptr resolve( const VariablesMap &, VariablesSet & ) const override {
            return shared_from_this();
        }

        bool resolved() const override { return true; }
        bool textPart() const override { return true; }
        Ptr assign(const Variables &) const override { return shared_from_this(); }
    };

    class Variable : public Part, public boost::enable_shared_from_this<Variable>  {
        std::string text_;
    public:
        Variable( const std::string & v ) : text_(v) {}

        void stream(std::string &) const override {
            throw std::logic_error("Trying to stream unresolved variable " + text());
        }

        const std::string & text() const override { return text_; }

        Ptr resolve( const VariablesMap & map, VariablesSet & used ) const override {
            VariablesMap::const_iterator i = map.find(text());
            if(i!=map.end()) {
                used.insert(i->first);
                std::ostringstream s;
                s << '$' << i->second;
                return boost::make_shared<Text>(s.str());
            }
            return shared_from_this();
        }

        bool resolved() const override { return false; }
        bool textPart() const override { return false; }
        Ptr assign(const Variables &) const override { return shared_from_this(); }
    };

    class ParameterResolved : public BodyPart,
                              public boost::enable_shared_from_this<ParameterResolved>  {
        std::string text_;
        std::size_t id_;
    public:
        ParameterResolved( const std::string & text, std::size_t id )
        : text_(text), id_(id) {}

        void stream(std::string &) const override {
            throw std::logic_error("Trying to stream parameter " + text() + "with no value set");
        }

        const std::string & text() const override { return text_; }
        std::size_t id() const { return id_; }

        Ptr resolve( const VariablesMap &, VariablesSet & ) const override {
            return shared_from_this();
        }
        bool resolved() const override { return true; }
        bool textPart() const override { return false; }
        Ptr assign( const Variables & v ) const override {
            if( id() >= v.size() ) {
                std::ostringstream s;
                s << "can not find value with id " << id()
                        << " for the parameter " << text();
                throw std::out_of_range(s.str());
            }
            return boost::make_shared<Text>(v[id()]);
        }
    };

    class Parameter : public BodyPart, public boost::enable_shared_from_this<Parameter>  {
        std::string text_;
    public:
        Parameter( const std::string & v ) : text_(v) {}

        void stream(std::string &) const override {
            throw std::logic_error("Trying to stream unresolved parameter " + text());
        }

        const std::string & text() const override { return text_; }

        Ptr resolve( const VariablesMap & map, VariablesSet & used) const override {
            VariablesMap::const_iterator i = map.find(text());
            if(i!=map.end()) {
                used.insert(i->first);
                return boost::make_shared<ParameterResolved>(text(), i->second);
            }
            return shared_from_this();
        }

        bool resolved() const override { return false; }
        bool textPart() const override { return false; }
        Ptr assign(const Variables &) const override { return shared_from_this(); }
    };

    void stream(std::string & s) const {
        std::for_each(begin(), end(), [&s](const Part & v) { v.stream(s); });
    }

    template <typename T>
    void add( const std::string & v ) {
        parts_.push_back(boost::make_shared<T>(v));
    }

    typedef std::vector<BodyPart::Ptr> Parts;
    typedef boost::indirect_iterator<Parts::const_iterator> const_iterator;
    typedef const_iterator iterator;

    iterator begin() const { return parts_.begin(); }
    iterator end() const { return parts_.end(); }
    bool empty() const { return parts_.empty(); }
    Parts::size_type size() const { return parts_.size(); }

    bool resolve( const VariablesMap & map, VariablesSet & used ) {
        transformParts( [&](Part::Ptr p) { return p->resolve(map, used); });
        return mergeTextPartsIfResolved();
    }

    bool assign( const Part::Variables & v ) {
        transformParts( [&](Part::Ptr p) { return p->assign(v); });
        return mergeTextPartsIfResolved();
    }

    bool resolved() const;
private:
    bool mergeTextPartsIfResolved() {
        if( !this->resolved() ) {
            return false;
        }
        Parts merged;
        mergeTextParts(parts_, std::back_inserter(merged));
        parts_ = std::move(merged);
        return true;
    }

    template <typename Pred>
    void transformParts( Pred pred ) {
        Parts transformed;
        for(auto & p : parts_) {
            transformed.push_back(pred(p));
        }
        parts_ = std::move(transformed);
    }

    template <typename Range>
    static Range getMergeableRange( const Range & r ) {
        using namespace boost;
        using namespace boost::range;

        typedef const typename Range::value_type & Cref;

        const auto m = find_if<return_found_end>(r,
                        [](Cref v) { return v->textPart(); });
        return find_if<return_begin_found>(m,
                        [](Cref v) { return !v->textPart(); });
    }

    template <typename Range>
    static std::size_t totalTextLength( const Range & r ) {
        std::size_t length(0);
        for( const auto & i : r ) {
            length += i->text().length();
        }
        return length;
    }

    template <typename Range>
    static std::string text( const Range & r ) {
        std::string retval;
        retval.reserve(totalTextLength(r));
        for( const auto & i : r ) {
            i->stream(retval);
        }
        return retval;
    }

    template <typename Range, typename Out>
    static void mergeTextParts(Range in, Out out) {
        using boost::make_iterator_range;

        auto r = make_iterator_range(in);
        while( !r.empty() ) {
            const auto mergeable = getMergeableRange( r ) ;
            const auto unmergeable = make_iterator_range(r.begin(), mergeable.begin());

            copy(unmergeable, out);

            if( !mergeable.empty() ) {
                *out++ = boost::make_shared<Text>(std::move(text(mergeable)));
            }
            r = make_iterator_range(mergeable.end(), r.end());
        }
    }

    Parts  parts_;
};


struct UnresolvedPart {
    bool operator()( const Body::Part & v ) const {
        return !v.resolved();
    }
};

inline bool Body::resolved() const {
    return std::find_if(begin(), end(), UnresolvedPart() ) == end();
}

inline auto getUnresolved(const Body & body) ->
        decltype(boost::adaptors::filter(body, UnresolvedPart())){
    return boost::adaptors::filter(body, UnresolvedPart());
}
} // namespace query
} // namespace pgg
