#pragma once

#include <pgg/query/query.h>
#include <pgg/query/helper.h>
#include <pgg/query/traits.h>
#include <pgg/query/repository/table.h>
#include <pgg/database/database.h>
#include <pgg/type_name.h>
#include <pgg/numeric_cast.h>
#include <tuple>
#include <type_traits>
#include <boost/hana/core/is_a.hpp>
#include <boost/hana/transform.hpp>
#include <boost/hana/tuple.hpp>
#include <boost/hana/unpack.hpp>

namespace pgg {
namespace query {

namespace details {

struct VariablesMapper {
    VariablesMap & map;
    VariablesMapper(VariablesMap & map) : map(map) {}

    template <typename Val>
    void mapValue( const Val&, const std::string & name ) const {
        auto res = map.insert(std::make_pair(name, newId()));
        if(!res.second) {
            std::ostringstream s;
            s << "Redefinition of the argument " << name
                    << "previous index is " << res.first->second;
            throw std::invalid_argument(s.str());
        }
    }

    std::size_t newId() const {
        return map.size()+1;
    }
};

struct ParameterMapper {
    VariablesMap & map;
    ParameterMapper(VariablesMap & map) : map(map) {}

    template <typename Param>
    void mapParameter( const Param & p ) const {
        auto res = map.insert(std::make_pair(p.name(), index()));
        if(!res.second) {
            std::ostringstream s;
            s << "Redefinition of the parameter " << p.name()
                    << "previous index is " << res.first->second;
            throw std::invalid_argument(s.str());
        }
    }

    std::size_t index() const { return map.size(); }
};

struct ParameterExtractor {
    typedef repository::ParametersTable Table;
    typedef std::vector<ParameterPart> Result;
    const Table & table;
    Result & v;

    ParameterExtractor(const Table & table, Result & v)
    : table(table), v(v) {}

    template <typename Param>
    void mapParameter( const Param & p ) const {
        v.push_back(table.value(p));
    }
};

template <typename T, typename MapperT>
struct HasMapMethod {
    template <typename U>
    static auto check(const U& u) -> decltype(u.map(std::declval<const MapperT&>()),
                                              std::true_type());

    static std::false_type check(...);

    static constexpr bool value = decltype(check(std::declval<const T>()))::value;
};

template <typename T, typename MapperT>
struct HasMapParameterMethod {
    template <typename U>
    static auto check(const U& u) -> decltype(u.mapParameter(std::declval<const MapperT&>()),
                                              std::true_type());

    static std::false_type check(...);

    static constexpr bool value = decltype(check(std::declval<const T>()))::value;
};

template<typename T, typename = std::void_t<>>
struct HasParameterGetMethod: std::false_type { };

template<typename T>
struct HasParameterGetMethod<T, std::void_t<
    decltype(std::declval<T>().get())>
>: std::true_type { };

template <typename... Args>
struct is_vector: std::false_type { };

template <typename... Args>
struct is_vector<std::vector<Args...>> : std::true_type { };

template<typename T>
inline decltype(auto) castUnsignedToSigned(T&& t) {
    using Type = std::decay_t<T>;
    if constexpr (is_vector<Type>::value) {
        using ValueType = typename Type::value_type;
        if constexpr (std::is_unsigned_v<ValueType>) {
            using SignedValueType = std::make_signed_t<ValueType>;
            auto cast = [](const auto& val) {
                return PGG_NUMERIC_CAST(SignedValueType, val);
            };
            auto caster = t | boost::adaptors::transformed(cast);
            using Allocator = typename std::allocator_traits<typename Type::allocator_type>::template rebind_alloc<SignedValueType>;
            return std::vector<SignedValueType, Allocator> {caster.begin(), caster.end()};
        } else {
            return std::forward<T>(t);
        }
    } else {
        if constexpr (std::is_unsigned_v<Type>) {
            return PGG_NUMERIC_CAST(std::make_signed_t<Type>, t);
        } else {
            return std::forward<T>(t);
        }
    }
}

} // namespace details

template<typename Base, typename... Args>
struct Composer;

template<typename Base, typename Head, typename... Tail>
struct Composer<Base, Head, Tail...> : Helper<Base, Head>, Composer<Base, Tail...> {
    typedef Helper<Base, Head> This;
    typedef Composer<Base, Tail...> Next;

    using This::set;
    using Next::set;

    template <typename MapperT>
    std::enable_if_t< details::HasMapMethod<This, MapperT>::value, void>
    map(const MapperT & m) const {
        This::map(m);
        Next::map(m);
    }
    template <typename MapperT>
    std::enable_if_t< !details::HasMapMethod<This, MapperT>::value, void>
    map(const MapperT & m) const {
        Next::map(m);
    }

    template <typename MapperT>
    std::enable_if_t< details::HasMapParameterMethod<This, MapperT>::value, void>
    mapParameter(const MapperT & m) const {
        This::mapParameter(m);
        Next::mapParameter(m);
    }
    template <typename MapperT>
    std::enable_if_t< !details::HasMapParameterMethod<This, MapperT>::value, void>
    mapParameter(const MapperT & m) const {
        Next::mapParameter(m);
    }

    template <typename... Args>
    auto makeParametersTuple(Args&&... args) const {
        if constexpr (details::HasParameterGetMethod<This>::value) {
            if constexpr (boost::hana::is_a<boost::hana::tuple_tag, decltype(This::get())>) {
                auto params = boost::hana::transform(This::get(), [](auto&& t) {
                        return details::castUnsignedToSigned(std::forward<decltype(t)>(t));
                    });
                return boost::hana::unpack(params, [&](auto&&... unpackedArgs) {
                    return Next::makeParametersTuple(std::forward<Args>(args)...,
                        std::forward<decltype(unpackedArgs)>(unpackedArgs)...);
                });
            } else {
                return Next::makeParametersTuple(std::forward<Args>(args)...,
                    details::castUnsignedToSigned(This::get()));
            }
        } else {
            return Next::makeParametersTuple(std::forward<Args>(args)...);
        }

    }
};


template<typename Base>
struct Composer<Base> {
    template <typename MapperT>
    void map(const MapperT & ) const {}
    template <typename MapperT>
    void mapParameter(const MapperT &) const {}
    void set() const {}
    template <typename... Args>
    auto makeParametersTuple(Args&&... args) const {
        return boost::hana::make_tuple(std::forward<Args>(args)...);
    }
};


inline std::string sanitaizeQueryComment(const std::string & c) {
    const auto isSafeChar = [](char ch) {
        return ch >= ' ' && ch <= '~' && (
            ch != '"' &&
            ch != '\'' &&
            ch != '*' &&
            ch != '-' &&
            ch != '/' &&
            ch != '\\'
        );
    };
    std::string retval;
    retval.reserve(c.size());
    std::copy_if(c.begin(), c.end(), std::back_inserter(retval), isSafeChar);
    return retval;
}

template <typename Base, typename... Args>
struct QueryImpl : public query::Query, query::Composer<Base, Args...> {
    typedef QueryImpl Inherited;
    typedef pgg::Database Connection;

    void request( const Connection & conn, Connection::RequestHandler handler ) const {
        conn.request(*this, handler);
    }
    void mapValues(const Mapper & m) const override {
        query::Composer<Base, Args...>::map(m);
    }
    Text text() const override {
        std::vector<std::string> params;
        this->mapParameter(
                details::ParameterExtractor(parametersTable(),params));
        auto body = this->traits().body;
        body.assign(params);
        Text text;
        body.stream(text);

        if( !comment().empty() ) {
            text = "/* " + comment() + " */" + text;
        }

        return text;
    }

    const query::Traits & traits() const {
        return traits_;
    }

    EndpointType endpointType() const override {
        return traits().options.endpoint;
    }


    void comment(std::string v) {
        comment_ = sanitaizeQueryComment(std::move(v));
    }

    const std::string & comment() const {
        return comment_;
    }

    static const std::string & queryName() {
        static const std::string retval = pgg::details::stripNamespaces(
                pgg::details::typeName<Base>());
        return retval;
    }

    const char * name() const override {
        return queryName().c_str();
    }

    template<typename TupleT, std::size_t Counter>
    struct Filler {
        static void fill( QueryImpl & q, TupleT&& args ) {
            q.set( std::get<Counter-1>(std::forward<TupleT>(args)) );
            Filler<TupleT, Counter-1>::fill( q, std::forward<TupleT>(args) );
        }
    };

    template<typename TupleT>
    struct Filler<TupleT, 0> {
        static void fill( QueryImpl &, TupleT&& ) {}
    };

    template <typename TupleT, typename... ArgsT>
    void fillWithTuple(TupleT && tuple, ArgsT&& ... args) {
        typedef typename std::remove_reference<TupleT>::type Tuple;
        Filler<TupleT, std::tuple_size<Tuple>::value>::fill( *this, std::forward<TupleT>(tuple) );
        fill(std::forward<ArgsT>(args)...);
    }

    template<typename... TupleArgs, typename... ArgsT>
    void fill(const std::tuple<TupleArgs...>& tuple, ArgsT&& ... args) {
        fillWithTuple( tuple, std::forward<ArgsT>(args)...);
    }

    template<typename... TupleArgs, typename... ArgsT>
    void fill(std::tuple<TupleArgs...>& tuple, ArgsT&& ... args) {
        fillWithTuple( tuple, std::forward<ArgsT>(args)...);
    }

    template<typename... TupleArgs, typename... ArgsT>
    void fill(std::tuple<TupleArgs...>&& tuple, ArgsT&& ... args) {
        fillWithTuple( std::move(tuple), std::forward<ArgsT>(args)...);
    }

    template<typename FirstArg, typename... ArgsT>
    void fill(FirstArg&& head, ArgsT&& ... args) {
        this->set(std::forward<FirstArg>(head));
        fill(std::forward<ArgsT>(args)...);
    }

    template<typename Arg>
    void fill(Arg&& arg) {
        this->set(std::forward<Arg>(arg));
    }

    void fill() {}

    Milliseconds timeout() const override {
        return traits().options.timeout;
    }

    bool debug() const override {
        return traits().options.debug;
    }

    typedef repository::ParametersTable ParametersTable;

    template<typename... ArgsT>
    QueryImpl( const query::Traits & traits,
            const ParametersTable & table,
            ArgsT&& ... args)
     : traits_(traits), parametersTable_(table) {
        fill(std::forward<ArgsT>(args)...);
    }

    static const VariablesMap & variablesMap() {
        static const VariablesMap retval(createVariablesMap());
        return retval;
    }

    static const VariablesMap & parametersMap() {
        static const VariablesMap retval(createParametersMap());
        return retval;
    }

    QueryPtr clone() const override {
        return boost::make_shared<Base>(static_cast<const Base&>(*this));
    }

    auto params() const {
        return query::Composer<Base, Args...>::makeParametersTuple();
    }

private:
    const ParametersTable & parametersTable() const {
        return parametersTable_;
    }
    std::string comment_;

    static const Base & nullObject() {
        static const query::Traits nullTraits;
        static const ParametersTable nullTable;
        static const Base retval(nullTraits, nullTable);
        return retval;
    }

    static VariablesMap createVariablesMap() {
        VariablesMap retval;
        nullObject().map(details::VariablesMapper(retval));
        return retval;
    }

    static VariablesMap createParametersMap() {
        VariablesMap retval;
        nullObject().mapParameter(details::ParameterMapper(retval));
        return retval;
    }
    const query::Traits & traits_;
    const ParametersTable & parametersTable_;
};

} // namespace query
} // namespace pgg
