#pragma once

#include <vector>
#include <string>
#include <ctime>
#include <list>
#include <chrono>
#include <pgg/enumeration.h>
#include <apq/query.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/optional.hpp>
#include <boost/range/algorithm.hpp>
#include <boost/range/algorithm/swap_ranges.hpp>
#include <boost/fusion/adapted/boost_tuple.hpp>
#include <boost/fusion/support/is_sequence.hpp>
#include <apq/serialize_composite.hpp>
#include <yamail/data/reflection.h>

namespace pgg {
namespace query {

namespace details {

template <typename T>
std::vector<T> allocVector( const std::size_t size ) {
    return std::vector<T>(size, T());
}

inline std::vector<std::string> allocStrVector( const std::size_t size ) {
    return allocVector<std::string>(size);
}

struct ToString {
    template <typename T>
    std::string operator()( const T & v ) const { return v.toString(); }
};

template <typename T, typename Enabled = void>
struct ValueMapper {
    static void map(apq::query & q, const T & v) {
        q.bind_const_string(std::move(boost::lexical_cast<std::string>(v)));
    }
};

template <typename T>
void mapValue(apq::query & q, T v) {
    ValueMapper<T>::map(q, std::move(v));
}

template <>
struct ValueMapper<int64_t> {
    static void map(apq::query & q, int64_t v) {
        q.bind_const_int64(v);
    }
};

template <>
struct ValueMapper<boost::posix_time::ptime> {
    static void map(apq::query & q, boost::posix_time::ptime v) {
        q.bind_const_string(boost::posix_time::to_simple_string(v) + " UTC");
    }
};

template <typename ... T>
struct ValueMapper<std::chrono::duration<T...>> {
    static void map(apq::query & q, std::chrono::duration<T...> v) {
        q.bind_const_string(yreflection::to_string(v));
    }
};

template <>
struct ValueMapper<double> {
    static void map(apq::query & q, double v) {
        q.bind_const_double(v);
    }
};

template <>
struct ValueMapper<std::string> {
    static void map(apq::query & q, std::string v) {
        q.bind_const_string(std::move(v));
    }
};

template <typename T>
struct ValueMapper<boost::optional<T>> {
    static void map(apq::query & q, boost::optional<T> v) {
        if (v) {
            mapValue(q, std::move(*v));
        } else {
            q.bind_null();
        }
    }
};

template <typename T>
struct ValueMapper<std::optional<T>> {
    static void map(apq::query & q, std::optional<T> v) {
        if (v) {
            mapValue(q, std::move(*v));
        } else {
            q.bind_null();
        }
    }
};

template <typename T>
struct ValueMapper<std::list<T>> {
    static void map(apq::query & q, std::list<T> l) {
        auto v = std::move(allocVector<T>(l.size()));
        mapValue(q, std::move(boost::swap_ranges( l, v )));
    }
};

template <>
struct ValueMapper<std::vector<int64_t>> {
    static void map(apq::query & q, std::vector<int64_t> v ) {
        q.bind_const_int64_vector(std::move(v));
    }
};

template <>
struct ValueMapper<std::vector<uint64_t>> {
    static void map(apq::query & q, std::vector<uint64_t> v ) {
        q.bind_const_int64_vector(std::move(v));
    }
};

template < typename T >
struct ValueMapper<std::vector<Enumeration<T>>> {
    static void map(apq::query & q, std::vector<Enumeration<T>> v ) {
        auto out = allocStrVector(v.size());
        boost::transform(v, out.begin(), ToString());
        mapValue(q, std::move(out));
    }
};

template <>
struct ValueMapper<std::vector<std::string>> {
    static void map(apq::query & q, std::vector<std::string> v ) {
        q.bind_const_string_vector(std::move(v));
    }
};

template <>
struct ValueMapper<boost::uuids::uuid> {
    static void map(apq::query& q, const boost::uuids::uuid uuid) {
        q.bind_const_uuid(uuid);
    }
};

template <>
struct ValueMapper<std::vector<boost::uuids::uuid>> {
    static void map(apq::query& q, const std::vector<boost::uuids::uuid>& uuids) {
        auto out = allocStrVector(uuids.size());
        boost::transform(uuids, out.begin(), &boost::uuids::to_string);
        mapValue(q, std::move(out));
    }
};

template <typename T>
struct ValueMapper<T, typename boost::enable_if<boost::fusion::traits::is_sequence<T>>::type> {
    static void map(apq::query & q, T v ) {
        mapValue(q, apq::serialize_composite<T>(v));
    }
};

template <typename T>
struct ValueMapper<std::vector<T>, typename boost::enable_if<boost::fusion::traits::is_sequence<T>>::type> {
    static void map(apq::query & q, std::vector<T> v ) {
        auto out = allocStrVector(v.size());
        boost::transform(v, out.begin(), apq::serialize_composite<T>);
        mapValue(q, out);
    }
};

} // namespace details

struct Mapper {
    template <typename T>
    void mapValue(T&& v, const char * /*name*/ ) const {
        details::mapValue(q, std::forward<T>(v));
    }

    template <typename T>
    void mapValue(T&& v, const std::string & name ) const {
        mapValue(std::forward<T>(v), name.c_str());
    }

    Mapper(apq::query & q) : q(q) {}
    apq::query & q;
};

} // namespace query
} // namespace pgg
