#pragma once

#include <pgg/query/helper.h>
#include <pgg/query/mapper.h>
#include <string>
#include <ctime>
#include <limits>
#include <boost/date_time/posix_time/posix_time.hpp>

namespace pgg {
namespace query {

template <typename Tag>
struct RangeTagTraits {
    typedef typename Tag::Value Value;
    typedef typename Tag::CRef CRef;

    static Value max() { return Tag::max(); }
    static Value min() { return Tag::min(); }
    static Value add( CRef a, CRef b ) { return Tag::add(a, b); }
};

template <typename Tag>
struct RangeWrapper {
    typedef RangeTagTraits<Tag> Traits;
    typedef typename Traits::Value Value;
    typedef typename Traits::CRef CRef;

    RangeWrapper() : from(Traits::min()), to(Traits::max()) {}
    RangeWrapper(CRef from, CRef to) : from(from), to(to) {}

    Value from;
    Value to;
};

template <typename T>
struct NumericRangeTag {
    typedef T Value;
    typedef Value CRef;
    static Value max() { return std::numeric_limits<Value>::max(); }
    static Value min() { return std::numeric_limits<Value>::min(); }
    static Value add( CRef a, CRef b ) { return a + b; }
};

struct DateRangeTag {
    typedef boost::posix_time::ptime Value;
    typedef const Value & CRef;
    static Value max() {
        return boost::posix_time::ptime(boost::gregorian::date(9999,1,1));
    }
    static Value min() {
        return boost::posix_time::from_time_t(0);
    }
};

struct DateRange : public RangeWrapper<DateRangeTag> {
    DateRange() {}
    DateRange( CRef from, CRef to )
    : RangeWrapper<DateRangeTag>(from, to) {}
    DateRange( std::time_t from )
    : RangeWrapper<DateRangeTag>(getValue(from), Traits::max()) {}
    DateRange( std::time_t from, std::time_t to )
    : RangeWrapper<DateRangeTag>(getValue(from), getValue(to)) {}

    static boost::posix_time::ptime getValue(std::time_t v) {
        return boost::posix_time::from_time_t(v);
    }
};

template<typename Base>
struct Helper<Base, DateRange> {
    typedef DateRange Range;

    template <typename MapperT>
    void map(const MapperT & m) const {
        m.mapValue(v_.from, "dateFrom");
        m.mapValue(v_.to, "dateTo");
    }

    void set(const Range & v) { v_ = v; }

    Base & dateFrom( Range::CRef v ) {
        v_.from = v;
        return static_cast<Base&>(*this);
    }
    Base & dateTo(Range::CRef v) {
        v_.to = v;
        return static_cast<Base&>(*this);
    }
    Base & dateFrom( std::time_t v ) { return dateFrom(DateRange::getValue(v)); }
    Base & dateTo( std::time_t v ) { return dateTo(DateRange::getValue(v)); }

    Base & dateIn(Range::CRef fromValue, Range::CRef toValue) {
        dateFrom(fromValue);
        return dateTo(toValue);
    }

    Base & dateIn( std::time_t from, std::time_t to ) {
        return dateIn(DateRange::getValue(from), DateRange::getValue(to));
    }
private:
    Range v_;
};


struct ImapIdRangeTag : public NumericRangeTag<std::size_t> {};
typedef RangeWrapper<ImapIdRangeTag> ImapIdRange;

template<typename Base>
struct Helper<Base, ImapIdRange> {
    typedef ImapIdRange Range;

    template <typename MapperT>
    void map(const MapperT & m) const {
        m.mapValue(v_.from, "imapIdFrom");
        m.mapValue(v_.to, "imapIdTo");
    }

    void set(const Range & v) { v_ = v; }

    Base& imapIdFrom(Range::CRef v) {
        v_.from = v;
        return static_cast<Base&>(*this);
    }

    Base& imapIdTo(Range::CRef v) {
        v_.to = v;
        return static_cast<Base&>(*this);
    }

    Base & imapIdRange(Range::CRef fromValue, Range::CRef toValue ) {
        imapIdFrom(fromValue);
        return imapIdTo(toValue);
    }

private:
    Range v_;
};


} // namespace query
} // namespace pgg
