#pragma once

#include <pgg/query/traits.h>
#include <pgg/query/body.h>
#include <pgg/query/query_register.h>
#include <pgg/query/repository/table.h>
#include <pgg/query/repository/linker.h>
#include <pgg/query/repository/parameters_resolver.h>
#include <pgg/share.h>
#include <sstream>

namespace pgg {
namespace query {

class Repository {
public:
    using QueryTable = repository::QueryTable;
    using ParametersTable = repository::ParametersTable;
    using ErrorCode = repository::ErrorCode;

    template <typename... Qs, typename... Ps>
    Repository(const TraitsMap & traits, const Parameters & parameters,
               RegisterQueries<Qs...>, RegisterParameters<Ps...>) {
        registerQueries<Qs...>();
        registerParameters<Ps...>();
        setTraits(traits, parameters);
    }

    void setTraits(const TraitsMap & traits, const Parameters & parameters) {
        ErrorCode error = parametersLinker.link(parametersTable_, parameters);
        if( error ) {
            handleParametersError(error);
        } else {
            error = linker.link(queryTable_, traits);
            if( error ) {
                handleQueryError(error);
            }
        }
    }

    template < typename QueryT, typename... ArgsT >
    QueryT query(ArgsT&& ... args) const {
        const auto i = queryTable().find<QueryT>();
        if( i==queryTable().end() ) {
            std::ostringstream s;
            s << "Query " << QueryT::queryName() << " was not registered";
            throw std::out_of_range(s.str());
        }
        return QueryT(i->traits(), parametersTable(), std::forward<ArgsT>(args)...);
    }
private:
    const QueryTable & queryTable() const { return queryTable_;}
    const ParametersTable & parametersTable() const { return parametersTable_; }
    static void handleParametersError(ErrorCode error) {
        throw std::logic_error( "error while processing parameters, " +
                error.category().message(error.value()) + ": " + error.message() );
    }
    static void handleQueryError(ErrorCode error) {
        throw std::logic_error( "error while processing queries, " +
                error.category().message(error.value()) + ": " + error.message() );
    }

    template <typename... Qs>
    void registerQueries() {
        const int fold[] = { (queryTable_.add<Qs>(), 0)..., 0 };
        (void)fold;
    }

    template <typename... Ps>
    void registerParameters() {
        const int fold[] = { (parametersTable_.add<Ps>(), 0)..., 0 };
        (void)fold;
    }

    QueryTable queryTable_;
    ParametersTable parametersTable_;
    const repository::Linker<repository::Resolver> linker;
    const repository::Linker<repository::ParametersResolver> parametersLinker;
};

typedef boost::shared_ptr<Repository> RepositoryPtr;

} // namespace query
} // namespace pgg
