#pragma once

#include <pgg/query/traits.h>
#include <pgg/query/body.h>
#include <pgg/query/repository/table.h>
#include <pgg/query/repository/resolver.h>
#include <pgg/query/repository/error.h>
#include <boost/range/algorithm.hpp>

namespace pgg {
namespace query {
namespace repository {

template <typename ResolverT>
class Linker {
public:
    Linker(const ResolverT & resolver = ResolverT()) : resolver(resolver) {}

    template <typename Table, typename TraitsMap>
    ErrorCode link(Table & table, const TraitsMap & traits) const {
        if( traits.size() < table.size() ) {
            return handleTraitsSizeError(table, traits);
        }
        for( auto & v : traits ) {
            if( const ErrorCode error = linkItem(v, table) ) {
                return error;
            }
        }
        return ErrorCode();
    }

private:
    template <typename Table, typename TraitsMap>
    ErrorCode handleTraitsSizeError(const Table & table, const TraitsMap & traits) const {
        std::vector<std::string> namesInTable;
        auto takeName = [](const typename Table::Item& item) { return item.name(); };
        std::transform(table.begin(), table.end(), std::back_inserter(namesInTable), takeName);

        std::vector<std::string> names;
        boost::set_difference( boost::sort(namesInTable), traits | boost::adaptors::map_keys,
                std::back_inserter(names));

        return ErrorCode(error::unspecified, boost::join(names, ", "));
    }

    template <typename Table, typename TraitsMapItem>
    ErrorCode linkItem(const TraitsMapItem & t, Table & table) const {
        const std::string & name = t.first;
        const auto & traits = t.second;
        const auto i = table.find(name);
        if(i == table.end()) {
            return ErrorCode(error::unregistered, name);
        }
        const ErrorCode err = resolver.resolve( *i, traits );
        if( err ) {
            return ErrorCode(err.base(), "in '" + name + "': " + err.message());
        }
        return ErrorCode();
    }
    const ResolverT resolver;
};

} // namespace repository
} // namespace query
} // namespace pgg
