#pragma once

#include <pgg/query/traits.h>
#include <pgg/query/parameter.h>
#include <pgg/query/repository/error.h>
#include <boost/range/algorithm.hpp>
#include <boost/algorithm/string/join.hpp>

namespace pgg {
namespace query {
namespace repository {

class ParametersResolver {
public:
    template <typename Item>
    ErrorCode resolve( Item & item, const ParameterMap & in ) const {
        VariablesSet used;
        auto traits = item.traits();

        for( const auto & i : in ) {
            const auto & name = i.first;
            try {
                const auto index = item.indexOfValue(name);
                const auto maxIndex = traits.size() - 1;
                if(index > maxIndex) {
                    std::ostringstream s;
                    s << "enum value " << name << "=" << index << " of parameter '"
                            << item.name() <<"' is out of range " << maxIndex;
                    return ErrorCode(error::internal, s.str());
                }
                traits[index] = i.second;used.insert(name);
            } catch( const std::out_of_range & ) {
                return ErrorCode(error::unresolved,
                        "unknown parameter value '" + name + "'" );
            }
        }

        if(used.size() != item.valueNames().size() ) {
            return handleUnused( item.valueNames(), used );
        }

        item.traits(std::move(traits));

        return ErrorCode();
    }

private:
    ErrorCode handleUnused( const ParameterValuesNames & all, const VariablesSet & used ) const {
        std::vector<std::string> names;
        boost::set_difference( all, used, std::back_inserter(names));
        return ErrorCode(error::unused, boost::join(names, ", "));
    }
};


} // namespace repository
} // namespace query
} // namespace pgg
