#pragma once

#include <pgg/query/traits.h>
#include <pgg/query/body.h>
#include <pgg/query/repository/error.h>
#include <boost/range/adaptor/map.hpp>
#include <boost/range/algorithm.hpp>
#include <boost/algorithm/string/join.hpp>
#include <sstream>

namespace pgg {
namespace query {
namespace repository {

class Resolver {
public:
    template <typename Item>
    ErrorCode resolve( Item & item, const Traits & in ) const {
        auto out = in;
        const auto err = resolve( item.variables(), item.parameters(), out.body );
        if(!err) {
            item.traits(out);
        }
        return err;
    }
    ErrorCode resolve( const VariablesMap & vars, const VariablesMap & params, Body & body ) const {
        VariablesSet used;
        VariablesMap all(vars);
        all.insert(params.begin(), params.end());
        if(!body.resolve(all, used)) {
            return handleUnresolved( body );
        }
        return checkUnused( all, used );
    }

private:
    ErrorCode handleUnresolved( const Body & body ) const {
        std::vector<std::string> names;
        auto unresolved = getUnresolved(body);
        const auto toText = [](const BodyPart& part) { return part.text(); };
        std::transform(boost::begin(unresolved), boost::end(unresolved), std::back_inserter(names), toText);

        return ErrorCode(error::unresolved, boost::join(names, ", "));
    }

    ErrorCode checkUnused( const VariablesMap & vars, const VariablesSet & used ) const {
        std::vector<std::string> names;
        boost::set_difference( vars | boost::adaptors::map_keys, used, std::back_inserter(names));
        if(names.empty()) {
            return ErrorCode();
        }
        return ErrorCode(error::unused, boost::join(names, ", "));
    }
};


} // namespace repository
} // namespace query
} // namespace pgg
