#pragma once

#include <boost/multi_index_container.hpp>
#include <boost/multi_index/hashed_index.hpp>
#include <boost/multi_index/member.hpp>
#include <boost/multi_index/mem_fun.hpp>
#include <boost/multi_index/tag.hpp>
#include <pgg/query/traits.h>
#include <pgg/query/body.h>
#include <pgg/query/parameter.h>
#include <boost/iterator/indirect_iterator.hpp>
#include <sstream>
#include <typeinfo>
#include <typeindex>

namespace pgg {
namespace query {
namespace repository {

template <typename ItemT>
class TableBase {
public:
    typedef typename ItemT::Id Id;
    typedef typename ItemT::Name Name;

    typedef typename ItemT::Interface Item;
    typedef std::unique_ptr<Item> ItemPtr;
    typedef boost::multi_index_container<
        ItemPtr,
        boost::multi_index::indexed_by<
            boost::multi_index::hashed_unique<
                boost::multi_index::tag<Name>,
                boost::multi_index::const_mem_fun<Item, const Name &, &Item::name>
            >,
            boost::multi_index::hashed_unique<
                boost::multi_index::tag<Id>,
                boost::multi_index::const_mem_fun<Item, Id, &Item::id>
            >
        >
    > Items;

    typedef boost::indirect_iterator<typename Items::const_iterator, const Item> const_iterator;
    typedef boost::indirect_iterator<typename Items::iterator> iterator;

    template < typename T >
    void add() {
        typedef typename ItemT::template Impl<T> Impl;
        ItemPtr item( new Impl() );
        const std::string name = item->name();
        if( !items.insert(std::move(item)).second ) {
            throw std::invalid_argument("TableBase::add(): duplicate item name: " + name);
        }
    }

    std::size_t size() const { return items.size(); }
    iterator begin() { return items.begin(); }
    iterator end() { return items.end(); }
    const_iterator begin() const { return items.begin(); }
    const_iterator end() const { return items.end(); }
    const_iterator cbegin() const { return begin(); }
    const_iterator cend() const { return end(); }
    const_iterator find( const Name & key ) const { return find( this, key ); }
    iterator find( const Name & key ) { return find( this, key ); }
    template <typename T>
    const_iterator find() const { return find( this, id<T>() ); }
    template <typename T>
    iterator find() { return find( this, id<T>() ); }
private:
    template <typename T>
    static Id id() { return ItemT::template id<T>(); }
    template <typename T, typename Self>
    static auto find( Self * self, const T & key ) -> decltype(self->begin()) {
        auto & items = self->items;
        return items.template project<0>(items.template get<T>().find( key ));
    }
    Items items;
};

struct ParameterTableEntry {
    typedef std::type_index Id;
    typedef std::string Name;

    template <typename T>
    static Id id() { return typeid(T); }

    class Interface {
        ParameterValues traits_;
    public:
        virtual ~Interface() {}
        virtual Id id() const = 0;
        virtual const Name & name() const = 0;
        virtual std::size_t indexOfValue( const std::string & name ) const = 0;
        virtual const ParameterValuesNames & valueNames() const = 0;
        const ParameterValues & traits() const { return traits_; }
        void traits( ParameterValues v ) { traits_ = std::move(v); }
    };

    template <typename T>
    class Impl : public Interface {
    public:
        Impl() {
            traits(ParameterValues(valueNames().size()));
        }
        Id id() const override { return ParameterTableEntry::id<T>(); }
        std::size_t indexOfValue( const std::string & name ) const override {
            return T::fromString(name);
        }
        const std::string & name() const override { return T::name(); }
        const ParameterValuesNames & valueNames() const override {
            return T::valueNames();
        }
    };
};

class ParametersTable : public TableBase<ParameterTableEntry> {
public:
    template< typename ParameterT >
    const ParameterPart & value( const ParameterT & p) const {
        auto i = find<ParameterT>();
        if( i == end() ) {
            std::ostringstream s;
            s << "Parameter " << ParameterT::name() << " was not registered";
            throw std::out_of_range(s.str());
        }
        return value(i->traits(), p);
    }
private:
    template <typename ParameterT>
    const ParameterPart & value(const ParameterValues & v, const ParameterT & p) const {
        const std::size_t valueIndex = p.value();
        if( valueIndex >= v.size() ) {
            std::ostringstream s;
            s << "Parameter " << ParameterT::name() << " has invalid value "
                    << p.value() << '(' << p.toString() << ')';
            throw std::out_of_range(s.str());
        }
        return v[valueIndex];
    }
};


struct QueryTableEntry {
    typedef std::type_index Id;
    typedef std::string Name;

    template <typename T>
    static Id id() { return typeid(T); }

    class Interface {
        Traits traits_;
    public:
        virtual ~Interface() {}
        virtual Id id() const = 0;
        virtual const Name & name() const = 0;
        virtual const VariablesMap & variables() const = 0;
        virtual const VariablesMap & parameters() const = 0;
        const Traits & traits() const { return traits_; };
        void traits( Traits v ) { traits_ = std::move(v); }
    };

    template <typename T>
    class Impl : public Interface {
    public:
        Id id() const override { return QueryTableEntry::id<T>(); }
        const Name & name() const override { return T::queryName(); }
        const VariablesMap & variables() const override { return T::variablesMap(); }
        const VariablesMap & parameters() const override { return T::parametersMap(); }
    };
};

typedef TableBase<QueryTableEntry> QueryTable;

} // namespace repository
} // namespace query
} // namespace pgg
