#pragma once

#include <memory>

#include <boost/make_shared.hpp>
#include <pgg/range.h>
#include <boost/enable_shared_from_this.hpp>
#include <pgg/query/transaction.h>
#include <pgg/share.h>

#include <boost/asio/coroutine.hpp>

namespace pgg {
namespace query {

using Coroutine = boost::asio::coroutine;

template <typename TransactionF, typename ConnProvider = ConnectionPtr,
        typename Transaction = query::Transaction>
class Transactional :
        public boost::enable_shared_from_this<
                Transactional<TransactionF, ConnProvider, Transaction>> {
public:
    using TransactionPtr = boost::shared_ptr<Transaction>;

    Transactional(ConnProvider c, TransactionPtr transaction, TransactionF f)
    : conn( c ), transaction( transaction ),
      f( std::move(f) ) {}

    void operator()() { f( coro, *this ); }

    template <typename RequestHandler>
    void request(const Query & q, RequestHandler&& handler) {
        transaction->request( q, wrapRequestHandler(std::forward<RequestHandler>(handler)));
    }
    template <typename FetchHandler>
    void fetch(const Query & q, FetchHandler&& handler) {
        transaction->fetch( q, wrapFetchHandler(std::forward<FetchHandler>(handler)));
    }
    template <typename UpdateHandler>
    void update(const Query & q, UpdateHandler&& handler) {
        transaction->update( q, wrapUpdateHandler(std::forward<UpdateHandler>(handler)));
    }
    template <typename ExecuteHandler>
    void execute(const Query & q, ExecuteHandler&& handler) {
        transaction->execute( q, wrapExecuteHandler(std::forward<ExecuteHandler>(handler)));
    }

    template <typename ExecuteHandler>
    void begin(ExecuteHandler&& handler, Milliseconds timeout) {
        transaction->begin( conn, wrapBeginHandler(std::forward<ExecuteHandler>(handler)), timeout);
    }
    template <typename ExecuteHandler>
    void commit(ExecuteHandler&& handler) {
        transaction->commit(wrapExecuteHandler(std::forward<ExecuteHandler>(handler)));
    }
    template <typename ExecuteHandler>
    void rollback(ExecuteHandler&& handler) {
        transaction->rollback(wrapExecuteHandler(std::forward<ExecuteHandler>(handler)));
    }

    bool inProgress() const noexcept { return transaction->inProgress(); }

    bool timedOut() const noexcept { return transaction->timedOut(); }

private:
    using error_code = database::error_code;
    template <typename ExecuteHandler>
    auto wrapBeginHandler(ExecuteHandler h) {
        return [h = std::move(h), self = share(this)](error_code ec, ConnectionPtr conn) {
            const bool succeeded = !ec;
            invokeWithConn(h, std::move(ec), std::move(conn));
            if (succeeded) {
                (*self)();
            }
        };
    }
    template <typename ExecuteHandler>
    auto wrapExecuteHandler(ExecuteHandler h) {
        return [h = std::move(h), thiz = share(this)](error_code ec) {
            const bool succeeded = !ec;
            h(ec);
            if(succeeded) {
                (*thiz)();
            }
        };
    }
    template <typename UpdateHandler>
    auto wrapUpdateHandler(UpdateHandler h) {
        return [h = std::move(h), thiz = share(this)](error_code ec, int i) {
            const bool succeeded = !ec;
            h(std::move(ec), i);
            if(succeeded) {
                (*thiz)();
            }
        };
    }
    template <typename FetchHandler>
    auto wrapFetchHandler(FetchHandler h) {
        return [h = std::move(h), thiz = share(this)](error_code ec, DataRange data) {
            const bool succeeded = !ec;
            h(std::move(ec), data);
            if(succeeded) {
                (*thiz)();
            }
        };
    }
    template <typename RequestHandler>
    auto wrapRequestHandler(RequestHandler h) {
        return [h = std::move(h), thiz = share(this)](error_code ec, apq::cursor c) {
            auto ctx = c.get();
            if(!ec && !ctx) {
                (*thiz)();
            } else {
                h(std::move(ec), {std::move(apq::row(*ctx)), std::move(apq::continuation(*ctx))});
            }
        };
    }

    Coroutine coro;
    ConnProvider conn;
    TransactionPtr transaction;
    TransactionF f;
};

template <typename T>
using TransactionalPtr = boost::shared_ptr<Transactional<T>>;

template <typename TransactionF>
TransactionalPtr<TransactionF> transactional(ConnectionPtr c, RepositoryPtr q, TransactionF f) {
    return boost::make_shared<Transactional<TransactionF>>(c, boost::make_shared<Transaction>(q), f);
}

template <typename TransactionF>
void runTransactional(ConnectionPtr c, RepositoryPtr q, TransactionF f) {
    ( *transactional(c, q, f) )();
}

namespace fallback {

template <typename TransactionF, typename DatabaseGenerator, typename Rule>
using Transactional = query::Transactional<TransactionF,
        DatabaseGenerator, Transaction<DatabaseGenerator, Rule>>;

template <typename T, typename D, typename R>
using TransactionalPtr = boost::shared_ptr<Transactional<T, D, R>>;

template <typename TransactionF, typename DatabaseGenerator, typename Rule>
TransactionalPtr<TransactionF, DatabaseGenerator, Rule> transactional(DatabaseGenerator c, RepositoryPtr q,
        TransactionF f, Rule rule) {
    return boost::make_shared<Transactional<TransactionF, DatabaseGenerator, Rule>>(std::move(c),
                boost::make_shared<Transaction<DatabaseGenerator, Rule>>(q, std::move(rule)), std::move(f));
}

template <typename TransactionF, typename DatabaseGenerator, typename Rule = fb::rules::Master>
void runTransactional(DatabaseGenerator c, RepositoryPtr q, TransactionF f,
        Rule rule = Rule()) {
    ( *transactional(std::move(c), q, std::move(f), std::move(rule)) )();
}

} // namespace fallback

} // namespace query
} // namespace pgg
