#pragma once

#include <pgg/query/query.h>
#include <pgg/query/transactional.h>
#include <pgg/query/repository.h>
#include <pgg/database/fallback.h>
#include <pgg/hooks/wrap.h>
#include <pgg/signature.h>
#include <io_result/io_result.h>

namespace pgg {

template <typename Query, typename Args>
struct QueryInfo {
    Args args;
};

namespace detail {

template <typename...>
struct make_void { using type = void;  };

template <typename... Ts>
using void_t = typename make_void<Ts...>::type;

template <typename T>
struct identity {
    using type = T;

    T operator()(T arg) const {
        return arg;
    }
};

template <typename T>
struct is_insert_iterator : std::false_type {};

template <typename C>
struct is_insert_iterator<std::back_insert_iterator<C>> : std::true_type {};

template <typename C>
struct is_insert_iterator<std::insert_iterator<C>> : std::true_type {};

template <typename T>
struct is_optional : std::false_type {};

template <typename T>
struct is_optional<io_result::Optional<T>> : std::true_type {};

template <typename T>
struct is_cursor : std::false_type {};

template <typename T>
struct is_cursor<io_result::Cursor<T>> : std::true_type {};

template <typename T, typename AlwaysVoid = void>
struct is_push_back_aware_container : std::false_type {};

template <typename T>
struct is_push_back_aware_container<T,
    void_t<decltype(std::declval<T&>().push_back(std::declval<typename T::value_type>()))>
> : std::true_type {};

template <typename T>
struct wrap_optional {
    using type = io_result::Optional<typename T::value_type>;
};

template <typename T>
struct wrap_cursor {
    using type = io_result::Sequence<typename T::value_type>;
};

template <typename ValueType>
struct default_converter {
    template <typename T>
    struct unpack_value_type {
        using type = typename T::value_type;
    };

    template <typename T>
    struct unpack {
        using type = typename std::conditional_t<
            is_push_back_aware_container<T>::value,
            unpack_value_type<T>,
            identity<T>
        >::type;
    };

    template <typename T>
    struct unpack<io_result::Optional<T>> {
        using type = T;
    };

    template <typename T>
    struct unpack<io_result::Cursor<T>> {
        using type = T;
    };

    template <typename T>
    struct unpack<io_result::Sequence<T>> {
        using type = T;
    };

    template <typename T>
    using unpack_t = typename unpack<T>::type;

    using type = identity<unpack_t<ValueType>>;
};

template <typename Handler>
using default_converter_by_handler =
    typename default_converter<signature::select_value_type<Handler>>::type;

template <typename HookValue>
using default_converter_by_hook_value = typename default_converter<HookValue>::type;

template <typename T>
struct is_query : std::false_type {};

template <>
struct is_query<query::Query> : std::true_type {};

template <typename Query, typename Args>
struct is_query<QueryInfo<Query, Args>> : std::true_type {};

} // namespace detail

template <typename Query, typename... Args>
auto withQuery(Args&&... args) {
    auto queryArgs = std::make_tuple(std::forward<Args>(args)...);
    return QueryInfo<Query, decltype(queryArgs)>{ std::move(queryArgs) };
}

template <typename DatabaseGenerator>
class BasicRequestExecutor {
    enum class InsertIteratorTag;
    enum class CallbackTag;
    enum class FetchCallbackTag;
    enum class RequestCallbackTag;
    enum class PureCallbackTag;
    enum class SequenceTag;
    enum class NonSequenceTag;

    template <typename Hook>
    using is_sequence = std::conditional_t<
        std::is_same<
            io_result::hooks::Strategy::type<Hook>,
            io_result::hooks::Strategy::Sequence
        >::value,
        SequenceTag,
        NonSequenceTag
    >;

    using Database = std::decay_t<decltype(*(std::declval<const DatabaseGenerator&>()()))>;
    using RequestCursor = signature::select_hook_value_type<typename Database::RequestHandler>;
    using FetchDataRange = signature::select_hook_value_type<typename Database::FetchHandler>;
public:
    BasicRequestExecutor(query::RepositoryPtr queryRepository, DatabaseGenerator db)
        : queryRepository(std::move(queryRepository)),
          db(std::move(db))
    {}

    // request

    template <typename HookValue,
              typename QueryType,
              typename Handler = io_result::sync_context,
              typename Converter = detail::default_converter_by_hook_value<HookValue>>
    auto request(QueryType&& query, Handler handler = io_result::use_sync,
                 Converter converter = Converter()) {
        using Hook = io_result::Hook<HookValue>;
        using Reflection = signature::reflection<Converter>;

        io_result::detail::init_async_result<Handler, Hook> init(handler);
        auto hook = wrapHook<Reflection, Hook>(init.handler, std::move(converter));
        dispatch(forwardQuery(std::forward<QueryType>(query)), std::move(hook), is_sequence<Hook>());
        return init.result.get();
    }

    template <typename QueryType,
              typename Handler,
              typename = std::enable_if_t<detail::is_query<std::decay_t<QueryType>>::value>,
              typename Converter = detail::default_converter_by_handler<Handler>>
    auto request(QueryType&& query, Handler handler, Converter converter = Converter()) {
        using Dispatch = std::conditional_t<
            detail::is_insert_iterator<Handler>::value,
            InsertIteratorTag,
            CallbackTag
        >;
        return requestImpl(forwardQuery(std::forward<QueryType>(query)), handler, std::move(converter),
                           Dispatch());
    }

    // update

    template <typename Handler = io_result::sync_context>
    auto update(const query::Query& query, Handler handler = io_result::use_sync) {
        io_result::detail::init_async_result<Handler, io_result::Hook<int>> init(handler);
        db()->update(query, init.handler);
        return init.result.get();
    }

    template <typename Query, typename Args, typename Handler = io_result::sync_context>
    auto update(QueryInfo<Query, Args>&& query, Handler handler = io_result::use_sync) {
        return update(makeQuery(std::move(query)), handler);
    }

    // execute

    template <typename Handler = io_result::sync_context>
    auto execute(const query::Query& query, Handler handler = io_result::use_sync) {
        io_result::detail::init_async_result<Handler, io_result::Hook<void>> init(handler);
        db()->execute(query, init.handler);
        return init.result.get();
    }

    template <typename Query, typename Args, typename Handler = io_result::sync_context>
    auto execute(QueryInfo<Query, Args>&& query, Handler handler = io_result::use_sync) {
        return execute(makeQuery(std::move(query)), handler);
    }

    // other

    template <typename TransactionF, typename Rule = fb::rules::Master>
    void runTransactional(TransactionF&& transaction, Rule rule = Rule()) const {
        query::fallback::runTransactional(db, queryRepository, std::forward<TransactionF>(transaction),
            std::move(rule))();
    }

private:
    template <typename Callback, typename Converter>
    auto requestImpl(const query::Query& query, Callback callback, Converter converter,
                     CallbackTag) {
        using Value = signature::select_hook_value_type<Callback>;
        using Dispatch = std::conditional_t<
            std::is_same<Value, FetchDataRange>::value,
            FetchCallbackTag,
            std::conditional_t<
                std::is_same<Value, RequestCursor>::value,
                RequestCallbackTag,
                PureCallbackTag
            >
        >;
        return requestImpl(query, callback, std::move(converter), Dispatch());
    }

    template <typename Callback, typename Converter>
    auto requestImpl(const query::Query& query, Callback callback, Converter converter,
                     PureCallbackTag) {
        using CallbackValue = signature::select_hook_value_type<Callback>;
        using HookValue = typename std::conditional_t<
            detail::is_optional<CallbackValue>::value,
            detail::wrap_optional<CallbackValue>,
            std::conditional_t<
                detail::is_cursor<CallbackValue>::value,
                detail::wrap_cursor<CallbackValue>,
                detail::identity<CallbackValue>
            >
        >::type;
        using Hook = io_result::Hook<HookValue>;
        using Reflection = signature::reflection<Converter>;

        io_result::detail::init_async_result<Callback, Hook> init(callback);
        dispatch(query, wrapHook<Reflection, Hook>(init.handler, std::move(converter)),
                 is_sequence<Hook>());
        return init.result.get();
    }

    template <typename Iterator, typename Converter>
    auto requestImpl(const query::Query& query, Iterator iter, Converter converter,
                     InsertIteratorTag) {
        using Reflection = signature::reflection<Converter>;
        using Value = std::decay_t<std::invoke_result_t<Converter, Reflection>>;
        using Hook = io_result::Hook<io_result::Sequence<Value>>;

        io_result::detail::init_async_result<Iterator, Hook> init(iter);
        auto hook = wrapHook<Reflection, Hook>(init.handler, std::move(converter));
        db()->request(query, std::move(hook));
        return init.result.get();
    }

    template <typename Callback, typename Converter>
    void requestImpl(const query::Query& query, Callback callback, Converter, FetchCallbackTag) {
        db()->fetch(query, std::move(callback));
    }

    template <typename Callback, typename Converter>
    void requestImpl(const query::Query& query, Callback callback, Converter, RequestCallbackTag) {
        db()->request(query, std::move(callback));
    }

    template <typename Handler>
    void dispatch(const query::Query& query, Handler handler, SequenceTag) {
        db()->request(query, handler);
    }

    template <typename Handler>
    void dispatch(const query::Query& query, Handler handler, NonSequenceTag) {
        db()->fetch(query, handler);
    }

    template <typename Query, typename Args>
    Query makeQuery(QueryInfo<Query, Args>&& queryHolder) {
        static_assert(std::is_base_of<query::Query, Query>::value,
                      "Query needs to be derived from pgg::query::Query");
        constexpr size_t argsCount = std::tuple_size<Args>::value;
        return makeQuery<Query>(std::move(queryHolder.args), std::make_index_sequence<argsCount>());
    }

    template <typename Query, typename... Args, size_t... Index>
    Query makeQuery(std::tuple<Args...>&& args, std::index_sequence<Index...>) {
        return queryRepository->query<Query>(std::get<Index>(std::move(args))...);
    }

    static const query::Query& forwardQuery(const query::Query& query) noexcept {
        return query;
    }

    template <typename Query, typename Args>
    Query forwardQuery(QueryInfo<Query, Args>&& query) {
        return makeQuery(std::move(query));
    }

    query::RepositoryPtr queryRepository;
    DatabaseGenerator db;
};

template <typename DatabaseGenerator>
BasicRequestExecutor<DatabaseGenerator> makeRequestExecutor(query::RepositoryPtr queryRepository,
                                                            DatabaseGenerator db) {
    return BasicRequestExecutor<DatabaseGenerator>(std::move(queryRepository), std::move(db));
}

using RequestExecutor = BasicRequestExecutor<fb::DatabaseGenerator>;

} // namespace pgg
