#pragma once

#include <pgg/error.h>
#include <pgg/query/traits.h>
#include <sharpei_client/sharpei_client.h>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/copy.hpp>


namespace pgg {

struct Credentials {
    std::string user;
    std::string password;

    static std::string concatOrEmpty( const char* key, const std::string& value) {
        return value.empty() ? "" : key + value;
    }

    std::string toString() const {
        return concatOrEmpty(" user=", user) + concatOrEmpty(" password=", password);
    }
};

struct SharpeiParams {
    sharpei::client::http::HttpClientPtr httpClient;
    sharpei::client::Settings settings;
    Credentials credentials;
};

using ConnStrings = std::vector<std::string>;
using OnResolve = io_result::Hook<ConnStrings>;
using OnShard = io_result::Hook<sharpei::client::Shard>;
using OnShardName = io_result::Hook<std::string>;

inline std::string makeSharpeiFullConnString(const std::string& connInfo,
                                      const std::string& credentials) {
    return connInfo + credentials;
}

struct MatchByEndpointType {
    using EndpointType = query::Traits::EndpointType;

    bool operator() (const sharpei::client::Shard::Database& db) const {
        switch (endpointType) {
            case EndpointType::master:
                return "master" == db.role;
            case EndpointType::replica:
                return "replica" == db.role;
            case EndpointType::lagReplica:
                return "replica" == db.role && 0 < db.state.lag;
            case EndpointType::noLagReplica:
                return "replica" == db.role && 0 == db.state.lag;
            case EndpointType::automatic:
                throw std::invalid_argument("Unexpected EndpointType "
                                            + endpointType.toString() + " in " + __PRETTY_FUNCTION__);
        }
    }

    EndpointType endpointType;
};

template <typename Params>
ConnStrings makeConnStrings(const Credentials& credentials, const Params& params,
        const sharpei::client::Shard& shard) {
    using boost::adaptors::filtered;
    using boost::adaptors::transformed;
    using sharpei::client::Shard;
    ConnStrings connStrings;
    boost::copy(shard.databases
        | filtered([] (const Shard::Database& db) {
            return db.status == "alive";
        })
        | filtered(MatchByEndpointType{params.endpointType_})
        | transformed([&] (const Shard::Database& db) {
            return makeSharpeiFullConnString(db.address.toString(), credentials.toString());
        }),
        std::back_inserter(connStrings));
    return connStrings;
}

template<typename Params>
OnShard getConnInfoHandler(OnResolve hook, Params p, Credentials credentials) {
    return [=] (error_code error, sharpei::client::Shard shard) {
            if (error) {
                hook(error, std::vector<std::string>());
            } else if (shard.databases.empty()) {
                hook(error_code(error::resolverTroubles, "answer is an empty array"),
                     std::vector<std::string>());
            } else {
                const auto connStrings = makeConnStrings<Params>(credentials, p, shard);
                if (connStrings.empty()) {
                    hook(error_code(error::noEndpointForRole,
                         "answer doesn't contains required databases for role " + p.endpointType_.toString()),
                         std::vector<std::string>());
                } else {
                    hook(error, std::move(connStrings));
                }
            }
        };
}

struct ShardInfo {
    using Endpoint = query::Traits::EndpointType;

    ShardInfo(std::string name, std::vector<Endpoint> aliveEndpoints)
            : name(std::move(name))
            , aliveEndpoints(std::move(aliveEndpoints)) {}

    explicit ShardInfo(std::string name)
            : ShardInfo(std::move(name), { Endpoint::master, Endpoint::replica, Endpoint::replica }) {}

    std::string name;
    std::vector<Endpoint> aliveEndpoints;
};

using OnShardInfo = io_result::Hook<ShardInfo>;

inline OnShard getShardInfoHandler(OnShardInfo hook) {
    return [=] (error_code error, sharpei::client::Shard shard) {
        if (error) {
            hook(error, ShardInfo(shard.name));
        } else {
            using boost::adaptors::filtered;
            using boost::adaptors::transformed;
            using sharpei::client::Shard;
            std::vector<ShardInfo::Endpoint> aliveEndpoints;
            boost::copy(shard.databases
                | filtered([] (const Shard::Database& db) {
                    return db.status == "alive";
                })
                | transformed([] (const Shard::Database& db) {
                    if (MatchByEndpointType{ShardInfo::Endpoint::master}(db)) {
                        return ShardInfo::Endpoint::master;
                    } else if (MatchByEndpointType{ShardInfo::Endpoint::replica}(db)) {
                        return ShardInfo::Endpoint::replica;
                    }
                    return ShardInfo::Endpoint::automatic;
                }),
                std::back_inserter(aliveEndpoints));
            hook(error, ShardInfo(shard.name, std::move(aliveEndpoints)));
        }
    };
}

} //namespace pgg
