#pragma once

#include <functional>
#include <pgg/request_info.h>
#include <pgg/service/resolver.h>
#include <pgg/query/traits.h>

namespace pgg {

struct UidResolveParams;

class UidResolver {
public:
    enum class UserType {
        existing,
        deleted,
        organization
    };

    using Id = std::string;
    using Params =  UidResolveParams;

    virtual ~UidResolver() {}
    virtual void asyncGetConnInfo(const UidResolveParams& params, OnResolve hook) const = 0;
    virtual void asyncGetShardName(const UidResolveParams& params, OnShardName hook) const = 0;
    virtual void asyncGetShardStatus(const UidResolveParams& params, OnShardInfo hook) const {
        asyncGetShardName(params, [h = std::move(hook)](error_code ec, std::string name){
            return h(std::move(ec), ShardInfo(std::move(name)));
        });
    }
};

struct UidResolveParams {
    using EndpointType = query::Traits::EndpointType;
    using Id = UidResolver::Id;
    Id userId_;
    EndpointType endpointType_;
    bool force_;
    UidResolveParams(const Id& u)
        : userId_(u), endpointType_(EndpointType::master), force_(false)
    { }

    UidResolveParams& endpointType(EndpointType v) { endpointType_ = v; return *this; }
    UidResolveParams& force(bool v) { force_ = v; return *this; }
};

using UidResolverPtr = std::shared_ptr<const UidResolver>;

class UidResolverFactory {
public:
    virtual ~UidResolverFactory() {}
    virtual UidResolverPtr product(Credentials credentials, const RequestInfo& requestInfo,
                                   UidResolver::UserType userType = UidResolver::UserType::existing) const = 0;
};

using UidResolverFactoryPtr = std::shared_ptr<const UidResolverFactory>;

UidResolverFactoryPtr createSharpeiUidResolverFactory(const SharpeiParams& params);
UidResolverFactoryPtr createFakeUidResolverFactory(const std::string& connString,
                                                   const std::string& shardName);

UidResolverPtr createSharpeiUidResolver(Credentials credentials, sharpei::client::SharpeiClientPtr client,
                                        UidResolver::UserType userType = UidResolver::UserType::existing);
UidResolverPtr createFakeUidResolver(const std::string& connStr);

} // namespace pgg
