#pragma once

#include <io_result/hooks.h>

#include <type_traits>
#include <tuple>

namespace pgg {
namespace signature {
namespace detail {

template <typename T>
struct has_call_operator {
    template <typename U>
    static std::true_type check(decltype(&U::operator()));

    template <typename>
    static std::false_type check(...);

    using value_type = decltype(check<T>(nullptr));
    static constexpr bool value = value_type::value;
};

template <typename T, bool HasCallOperator = true>
struct function_object_sig_unpacker {
    template <typename C, typename Res, typename... Args>
    static std::tuple<Args...> unpack(Res(C::*)(Args...));

    template <typename C, typename Res, typename... Args>
    static std::tuple<Args...> unpack(Res(C::*)(Args...) const);

    using args = decltype(unpack<T>(&T::operator()));
};

template <typename...>
struct false_t {
    static constexpr bool value = false;
};

template <typename T>
struct function_object_sig_unpacker<T, false> {
    static_assert(false_t<T>::value, "Can't infer hook value type");
};

template <typename T>
struct sig_unpacker : function_object_sig_unpacker<T, has_call_operator<T>::value>
{};

template <typename Res, typename... Args>
struct sig_unpacker<Res(*)(Args...)> {
    using args = std::tuple<Args...>;
};

template <typename T>
using sig_unpacker_args = typename sig_unpacker<T>::args;

template <typename T, size_t Index>
using get_nth_arg = std::tuple_element_t<Index, sig_unpacker_args<T>>;

template <typename ValueType>
struct select {
    using type = ValueType;
};

template <typename T>
struct select<io_result::hooks::detail::Cursor<T>> {
    using type = io_result::Sequence<T>;
};

template <typename T>
struct extract_type {
    using type = T;
};

template <typename T>
struct extract_type<io_result::Optional<T>> {
    using type = T;
};

template <typename T>
struct extract_type<io_result::Sequence<T>> {
    using type = T;
};

} // namespace detail

template <typename T>
using extract_type = typename detail::extract_type<T>::type;

template <typename Handler>
struct Selector {
    using value_type = detail::get_nth_arg<Handler, 1>;
    using hook_value_type = typename detail::select<value_type>::type;
};

template <typename T>
struct Selector<io_result::Hook<T>> {
    using value_type = extract_type<T>;
    using hook_value_type = T;
};

template <typename C>
struct Selector<std::back_insert_iterator<C>> {
    using value_type = typename C::value_type;
    using hook_value_type = io_result::Sequence<value_type>;
};

template <typename C>
struct Selector<std::insert_iterator<C>> {
    using value_type = typename C::value_type;
    using hook_value_type = io_result::Sequence<value_type>;
};

template <typename Handler>
using select_hook_value_type = typename Selector<Handler>::hook_value_type;

template <typename Handler>
using select_value_type = typename Selector<Handler>::value_type;

template <typename Handler>
using select_hook_type = io_result::Hook<select_hook_value_type<Handler>>;

template <typename Converter>
using reflection = std::decay_t<detail::get_nth_arg<Converter, 0>>;

} // namespace signature
} // namespace pgg
