#include <pgg/service/shard_resolver.h>

namespace pgg {

class ShardResolverImpl: public ShardResolver {
public:
    ShardResolverImpl(Credentials credentials, sharpei::client::SharpeiClientPtr client)
        : credentials(std::move(credentials)), client(std::move(client))
    {}

    void asyncGetConnInfo(const ShardResolveParams& p, OnResolve hook) const override {
        using namespace sharpei::client;
        const auto credentials = this->credentials;
        const auto handler = getConnInfoHandler<ShardResolveParams>(hook, p, credentials);
        asyncGetConnInfoImpl(p, handler);
    }

    void asyncGetShardName(const ShardResolveParams& p, OnShardName hook) const override {
        using sharpei::client::Shard;
        const auto handler = [=] (error_code error, Shard shard) {
            hook(error, std::move(shard.name));
        };
        asyncGetConnInfoImpl(p, handler);
    }

    void asyncGetShardStatus(const ShardResolveParams& p, OnShardInfo hook) const override {
        const auto handler = getShardInfoHandler(hook);
        asyncGetConnInfoImpl(p, handler);
    }

private:
    Credentials credentials;
    sharpei::client::SharpeiClientPtr client;

    template <class Handler>
    void asyncGetConnInfoImpl(const ShardResolveParams& p, Handler handler) const {
        client->asyncStatById(p.shardId_, handler);
    }
};

class ShardResolverFactoryImpl: public ShardResolverFactory {
public:
    ShardResolverFactoryImpl(const SharpeiParams& params)
        : sharpeiParams_(params)
    {}

    ShardResolverPtr product(Credentials credentials, const RequestInfo& requestInfo) const override {
        const sharpei::client::RequestInfo reqInfo {requestInfo.requestId, requestInfo.connectionId,
            requestInfo.clientType, requestInfo.userIp, requestInfo.requestId};
        using namespace sharpei::client;
        const auto sharpeiClient = createSharpeiClient(sharpeiParams_.httpClient, sharpeiParams_.settings,
            reqInfo);
        return std::make_shared<ShardResolverImpl>(std::move(credentials), makeCached(sharpeiClient));
    }

private:
    SharpeiParams sharpeiParams_;
};

ShardResolverFactoryPtr createShardResolverFactory(const SharpeiParams& params) {
    return std::make_shared<ShardResolverFactoryImpl>(params);
}

ShardResolverPtr createShardResolver(Credentials credentials, sharpei::client::SharpeiClientPtr client) {
    return std::make_shared<ShardResolverImpl>(std::move(credentials), std::move(client));
}


class FakeShardResolver: public ShardResolver, public ShardResolverFactory {
public:
    FakeShardResolver(const std::string& connStr, const std::string& shardName)
        : connStr_(connStr), shardName_(shardName)
    {}

    void asyncGetConnInfo(const ShardResolveParams&, OnResolve hook) const override {
        hook(error_code(), {connStr_});
    }

    void asyncGetShardName(const ShardResolveParams&, OnShardName hook) const override {
        hook(error_code(), shardName_);
    }

    ShardResolverPtr product(Credentials credentials, const RequestInfo&) const override {
        return std::make_shared<FakeShardResolver>(connStr_ + credentials.toString(), shardName_);
    }

private:
    std::string connStr_;
    std::string shardName_;
};

ShardResolverFactoryPtr createFakeShardResolverFactory(const std::string& connStr, const std::string& shardName) {
    return std::make_shared<FakeShardResolver>(connStr, shardName);
}

} //namespace pgg
