#include <backend/backend_types.h>

#include <cstdlib>
#include <boost/algorithm/string.hpp>
#include <boost/date_time/posix_time/posix_time_types.hpp>

namespace yimap {

AppendOptions AppendOptions::create(const yplatform::ptree& append)
{
    AppendOptions options;

    options.smtpgatePercent = append.get("smtpgate_percent", 0u);

    options.smtpHost = append.get("smtp.host", "");
    options.smtpPort = append.get("smtp.port", 1234);
    options.smtpTimeout = append.get("smtp.timeout", 10);
    options.smtpRetryCount = append.get("smtp.retry_count", 2);
    options.smtpIpv6 = append.get("smtp.ipv6", 0);

    options.httpHost = append.get("http.host", "");
    options.httpPort = append.get("http.port", 2443);
    options.httpRetryCount = append.get("http.retry_count", 2);
    options.httpRetrySmtp = append.get("http.retry_smtp", 1);

    return options;
}

bool AppendOptions::shouldUseSmtpgate(const std::string& sessionId) const
{
    std::hash<std::string> strHash;
    auto pseudoRand = strHash(sessionId) % 99;
    return !httpHost.empty() && smtpgatePercent > pseudoRand;
}

MbodyStorageOptions::BackendType MbodyStorageOptions::getBackendType(
    const std::string& sessionId,
    bool haveMimeParts,
    time_t receiveDate) const
{
    if (haveMimeParts && processNonEmptyMimesByMDS)
    {
        return BackendType::MDS;
    }
    if (boost::posix_time::from_time_t(receiveDate).date().year() < processByMulcaDateYear)
    {
        return BackendType::Mulca;
    }
    std::hash<std::string> strHash;
    auto pseudoRand = strHash(sessionId) % 99;
    if (pseudoRand < bothMulcaPercent)
    {
        return BackendType::BothMulca;
    }
    else if (pseudoRand < bothMulcaPercent + bothMDSPercent)
    {
        return BackendType::BothMDS;
    }
    else if (pseudoRand < bothMulcaPercent + bothMDSPercent + mdsPercent)
    {
        return BackendType::MDS;
    }
    else
    {
        return BackendType::Mulca;
    }
}

} // namespace yimap
