#include <yplatform/log.h>
#include <yplatform/module.h>
#include <yplatform/task_context.h>
#include <yplatform/find.h>
#include <yplatform/net/handlers/timer_handler.h>

#include <common/context.h>

#include <macs/hooks.h>
#include <macs_pg/macs_pg.h>
#include <macs_pg/integration/pa_profiler.h>

#include <backend/sharpei_resolver.hpp>

#include "backend_session.h"

#include <boost/algorithm/string/replace.hpp>
#include <pa/async.h>
#include <macs/user_journal.h>
#include <user_journal/parameters/parameter_id.h>
#include <user_journal/logging.h>
#include <user_journal/service.h>
#include <user_journal/service_factory.h>

using namespace user_journal;

namespace yimap { namespace backend {

namespace {

template <typename Duration>
auto toFloatSeconds(const Duration& d)
{
    return yplatform::time_traits::duration_cast<yplatform::time_traits::float_seconds>(d).count();
}

}

class SharpeiHttp;

class UserJournalLogger : public logging::Log
{
public:
    void warning(const std::string&, const std::string&, const std::string&, int) override
    {
    }
    void error(const std::string&, const std::string&, const std::string&, int) override
    {
    }
    void notice(const std::string&, const std::string&, const std::string&) override
    {
    }
    void debug(const std::string&, const std::string&, const std::string&) override
    {
    }
};

class MacsLogger : public macs::pg::logging::Log
{
public:
    MacsLogger(const ShortSessionInfo& sessionInfo) : sessionInfo(sessionInfo)
    {
    }

    void warning(const macs::pg::logging::Method& method, const macs::pg::logging::Message& message)
        override
    {
        L_(error) << sessionInfo.logPrefix() << "Macs warning from " << method << ": "
                  << boost::replace_all_copy(message, "\n", "\\n");
    }
    void error(const macs::pg::logging::Method& method, const macs::pg::logging::Message& message)
        override
    {
        L_(error) << sessionInfo.logPrefix() << "Macs error from " << method << ": "
                  << boost::replace_all_copy(message, "\n", "\\n");
    }
    void notice(const macs::pg::logging::Method& method, const macs::pg::logging::Message& message)
        override
    {
        L_(error) << sessionInfo.logPrefix() << "Macs notice from " << method << ": "
                  << boost::replace_all_copy(message, "\n", "\\n");
    }
    void debug(const macs::pg::logging::Method& method, const macs::pg::logging::Message& message)
        override
    {
        L_(debug) << sessionInfo.logPrefix() << "Macs debug from " << method << ": "
                  << boost::replace_all_copy(message, "\n", "\\n");
    }

protected:
    ShortSessionInfo sessionInfo;
};

//-----------------------------------------------------------------------------
// Backend service to manage DB and backend requests

class BackendServiceImpl
    : public yplatform::module
    , public BackendService
{
public:
    BackendServiceImpl();
    virtual ~BackendServiceImpl();

    virtual void init(const yplatform::ptree& xml);
    virtual void fini();
    virtual void start()
    {
    }
    virtual void stop()
    {
    }

    yplatform::ptree get_stats() const override;

    void writeTskv(const std::string& record);

    UserJournalPtr createJournal(
        const std::string& uid,
        const std::string& clientIp,
        const std::string& mdb,
        const std::string& suid,
        const std::string& service) override;
    PgServicePtr createPgService(
        const ShortSessionInfo& sessionInfo,
        bool replica,
        const std::string& service) override;

protected:
    void initLogger(const yplatform::ptree& xml);
    void initUserJournal(const yplatform::ptree& xml);
    void initPg(const yplatform::ptree& xml);
    void readSharpeiSettings(const yplatform::ptree& xml);
    user_journal::Journal journal(
        const std::string& uid,
        const std::string& clientIp,
        const std::string& service,
        const std::string& mdb,
        const std::string& suid);

private:
    ::macs::pg::ConnectionPoolPtr pgPool_;
    ::macs::pg::QueryConf queryConf;
    ::macs::pg::SharpeiParams sharpeiParams;
    ::macs::pg::ServiceFactory::Milliseconds pqTransactionTimeout;
    bool pgProfilerEnabled = false;

    std::shared_ptr<user_journal::Service> userJournalService;
    boost::shared_ptr<yplatform::log::source> tskvLog;

    boost::asio::io_service ioService;
    boost::thread_group threadPool;
    std::unique_ptr<boost::asio::io_service::work> ioWork;
};

BackendServiceImpl::BackendServiceImpl()
{
    L_(debug) << "BackendServiceImpl task instaniated";
}

BackendServiceImpl::~BackendServiceImpl()
{
    L_(debug) << "BackendServiceImpl task destroyed";
}

void BackendServiceImpl::init(const yplatform::ptree& xml)
{
    const auto nthreads = 5u;
    ioWork.reset(new boost::asio::io_service::work(ioService));
    for (size_t i = 0; i < nthreads; ++i)
        threadPool.create_thread(boost::bind(&asio::io_service::run, &ioService));

    if (xml.get("pa.<xmlattr>.init", 0))
    {
        std::size_t max_size = xml.get("pa.<xmlattr>.max", 1000000);
        std::size_t dump_size = xml.get("pa.<xmlattr>.dump", 4096);
        std::string path = xml.get("pa.<xmlattr>.path", "");
        pa::async_profiler::init(max_size, dump_size, path);
    }

    initLogger(xml);
    initUserJournal(xml);
    initPg(xml);
}

void BackendServiceImpl::initLogger(const yplatform::ptree& xml)
{
    if (xml.get("write_tskv", 0))
    {
        tskvLog.reset(new yplatform::log::source(YGLOBAL_LOG_SERVICE, "user_journal_tskv"));
    }
}

void BackendServiceImpl::initUserJournal(const yplatform::ptree& xml)
{
    user_journal::ServiceFactory factory;

    factory.logger(boost::make_shared<UserJournalLogger>());
    factory.fileWriter(boost::bind(&BackendServiceImpl::writeTskv, this, _1));
    factory.tableName(xml.get("user_journal.table_name", "users_history"));
    factory.tskvFormat(xml.get("user_journal.tskv_format", "mail-user-journal-log"));

    userJournalService = factory.product();
}

void BackendServiceImpl::initPg(const yplatform::ptree& xml)
{
    auto ioService = yplatform::find_reactor(xml.get<std::string>("pg.reactor"))->io();
    pgPool_ = ::macs::pg::ConnectionPoolFactory()
                  .ioService(*ioService)
                  .connectTimeout(macs::pg::Milliseconds(xml.get("pg.connect_timeout", 100)))
                  .queueTimeout(macs::pg::Milliseconds(xml.get("pg.queue_timeout", 100)))
                  .queryTimeout(macs::pg::Milliseconds(xml.get("pg.query_timeout", 15000)))
                  .maxConnections(xml.get("pg.max_connections", 30))
                  .product();
    pqTransactionTimeout =
        ::macs::pg::ServiceFactory::Milliseconds(xml.get("pg.transaction_timeout", 30000));
    pgProfilerEnabled = xml.get("pg.log_pa", false);

    sharpeiParams = readSharpeiParams(xml);
    queryConf = ::macs::pg::readQueryConfFile(xml.get<std::string>("query_conf"));
}

void BackendServiceImpl::writeTskv(const std::string& record)
{
    if (tskvLog)
    {
        auto& logger = *tskvLog;
        YLOG(logger, info) << record;
    }
}

void BackendServiceImpl::fini()
{
    pgPool_.reset();

    ioWork.reset();
    ioService.stop();
    threadPool.join_all();
}

yplatform::ptree BackendServiceImpl::get_stats() const
{
    auto pool_stats = pgg::statistic::getPoolStats(*pgPool_);
    yplatform::ptree pool_stats_node;
    for (auto& pair : pool_stats)
    {
        auto& conn_id = pair.first;
        auto& pool_stat = pair.second;

        yplatform::ptree stats_node;
        stats_node.put("free_connections", pool_stat.freeConnections);
        stats_node.put("busy_connections", pool_stat.busyConnections);
        stats_node.put("pending_connections", pool_stat.pendingConnections);
        stats_node.put("max_connections", pool_stat.maxConnections);
        stats_node.put("queue_size", pool_stat.queueSize);
        stats_node.put(
            "average_request_roundtrip", toFloatSeconds(pool_stat.averageRequestRoundtrip));
        stats_node.put(
            "average_request_db_latency", toFloatSeconds(pool_stat.averageRequestDbLatency));
        stats_node.put("average_wait_time", toFloatSeconds(pool_stat.averageWaitTime));
        stats_node.put("dropped_timeout", pool_stat.droppedTimedOut);
        stats_node.put("dropped_failed", pool_stat.droppedFailed);
        stats_node.put("dropped_busy", pool_stat.droppedBusy);
        stats_node.put("dropped_with_result", pool_stat.droppedWithResult);
        pool_stats_node.push_back(std::make_pair(conn_id, stats_node));
    }
    yplatform::ptree stats;
    stats.put_child("pool_stats", pool_stats_node);
    return stats;
}

//-----------------------------------------------------------------------------
// PG service

PgServicePtr BackendServiceImpl::createPgService(
    const ShortSessionInfo& sessionInfo,
    bool replica,
    const std::string& service)
{
    auto uidResolverFactory = macs::pg::createSharpeiUidResolverFactory(sharpeiParams);

    macs::pg::RequestInfo requestInfo;
    requestInfo.userIp = sessionInfo.clientIp;
    requestInfo.clientType = "imap";

    auto pgFactory = ::macs::pg::createSeviceFactory(pgPool_, uidResolverFactory);
    pgFactory->queryConf(queryConf);
    pgFactory->credentials(sharpeiParams.credentials);
    pgFactory->transactionTimeout(pqTransactionTimeout);
    if (pgProfilerEnabled)
    {
        pgFactory->profiler(macs::pg::integration::PaProfiler::create());
    }
    if (userJournalService)
    {
        pgFactory->userJournal(
            journal(sessionInfo.uid, sessionInfo.clientIp, service, sessionInfo.storage, ""));
    }
    pgFactory->logger(boost::make_shared<MacsLogger>(sessionInfo));
    pgFactory->autoQueryHandleStartegy(
        replica ? ::macs::pg::readReplicaThenMaster : ::macs::pg::readMasterThenReplica);
    pgFactory->requestInfo(std::move(requestInfo));

    return pgFactory->product(sessionInfo.uid);
}

//-----------------------------------------------------------------------------
// User journal logging

class ImapUserJournal : public macs::UserJournal
{
public:
    using Base = macs::UserJournal;

    ImapUserJournal(Base::Journal journal, std::string mdb)
        : Base(std::move(journal)), mdb(std::move(mdb))
    {
    }

private:
    std::string mdb;

    void asyncGetShardName(OnStringReceive handler) const override
    {
        handler(mdb);
    }
};

UserJournalPtr BackendServiceImpl::createJournal(
    const std::string& uid,
    const std::string& clientIp,
    const std::string& mdb,
    const std::string& suid,
    const std::string& service)
{
    return std::make_shared<ImapUserJournal>(journal(uid, clientIp, service, mdb, suid), mdb);
}

user_journal::Journal BackendServiceImpl::journal(
    const std::string& uid,
    const std::string& clientIp,
    const std::string& service,
    const std::string& mdb,
    const std::string& suid)
{
    if (!userJournalService)
    {
        throw std::runtime_error("UserJournalService is't initialized");
    }

    namespace id = user_journal::parameters::id;

    return userJournalService->createJournal(
        uid, id::module(service), id::ip(clientIp), id::suid(suid));
}

} // namespace backend
} // namespace yimap

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(yimap::backend::BackendServiceImpl)
