#include "message_access.h"

#include "storage/mulcagate_storage.h"
#include "util/header_parser.h"
#include "util/meta_parser.h"
#include "util/utfizer.h"

#include <yplatform/log.h>
#include <pa/stack.h>
#include <butil/butil.h>
#include <mimeparser/rfc2047.h>
#include <mimeparser/parameters.h>

#include <boost/algorithm/string.hpp>

namespace yimap { namespace mbody {

static const std::string WHOLE_MESSAGE_HID = "0";
const int MessageAccessLibmulca::inlineLevel_(15);

string extractMulcaHost(const string& stid)
{
    return stid.substr(0, stid.find('.'));
}

MessageAccessLibmulca::MessageAccessLibmulca(std::shared_ptr<Storage> storage)
    : storage(storage), headerText(*storage, *storage)
{
}

MetaAttributesPtr MessageAccessLibmulca::getHeaderStruct(const string& hid)
{
    if (hid == WHOLE_MESSAGE_HID)
    {
        MetaAttributesPtr messageHeadersParsed = getMessageHeaderParsed(rootHid);
        string subject = (*messageHeadersParsed)["subject"];

        string charset = "utf-8";
        subject = mulca_mime::decode_rfc2047(subject, charset);
        UTFizer::process(charset, subject);

        MetaAttributesPtr result(new MetaAttributes());
        (*result)["content_type.type"] = "message";
        (*result)["content_type.subtype"] = "rfc822";

        if (subject.empty()) subject = "No subject";

        (*result)["content_disposition.filename"] = subject + ".txt";

        return result;
    }

    if (meta_.empty())
        if (setMeta()) return NULL;

    MetaParts::iterator mpi;
    if ((mpi = meta_.find(hid)) != meta_.end())
    {
        MetaAttributesPtr res(new MetaAttributes(mpi->second));
        (*res)["hid"] = hid;
        return res;
    }
    return getHeaderStructInlineMessage(hid);
}

MetaLevelPtr MessageAccessLibmulca::getBodyStruct(const string& hid)
{
    if (meta_.empty())
        if (setMeta()) return NULL;

    MetaLevelPtr res(new MetaLevel);
    int i;
    for (i = 1;; i++)
    {
        std::ostringstream wrk;
        wrk << hid << "." << i;
        string subhid = wrk.str();
        if (meta_.find(subhid) != meta_.end()) res->insert(res->end(), subhid);
        else
            break;
    }

    if (i == 1)
    {
        InlineMessages::iterator imi = inlm_.find(hid);
        if (imi != inlm_.end())
        {
            res->insert(res->end(), hid + ".1");
        }
        else
        {
            MetaLevelPtr inlineRes = getBodyStructInlineMessage(hid);
            if (inlineRes)
            {
                res = inlineRes;
            }
        }
    }

    if (res->empty())
    {
        return res;
    }

    return res;
}

ConstStringPtr MessageAccessLibmulca::getHeader(const string& hid)
{
    if (hid != rootHid)
    {
        if (meta_.empty())
            if (setMetaFromHeaderOnly()) return NULL;

        MetaParts::iterator mpi;
        mpi = meta_.find(hid);
        if (mpi != meta_.end())
        {

            ConstStringPtr whole = getWhole();
            if (!whole) return NULL;
            string::size_type e = 0, b = 0;
            while (true)
            {
                e = boost::lexical_cast<string::size_type>(mpi->second["offset"]);
                if (!e) break;
                b = whole->rfind("\n--", (e - 2));
                if (b == string::npos)
                {
                    b = 0;
                    break;
                }
                b = whole->find('\n', ++b);
                if (b == string::npos)
                {
                    b = 0;
                    break;
                }
                ++b;
                break;
            }
            if (!e || !b || b >= e)
            {
                // can't be
                // whole->erase();
                whole.reset(new string());
            }
            else
            {
                ConstStringPtr substr(new string(whole->substr(b, e - b)));
                whole = substr;
            }
            return whole;
        }
        else
        {
            return getHeaderInlineMessage(hid);
        }
    }

    if (!headerText.fetchHeader())
    {
        setMetaIfNotEmpty(headerText.getMeta());

        return ConstStringPtr(new string(headerText.getHeader()));
    }

    return NULL;
}

ConstStringPtr MessageAccessLibmulca::getPart(const string& hid)
{
    pa::async_stack_profiler prof(pa::mulca);
    prof.set_host(extractMulcaHost(stid));
    prof.set_request("get_part");

    string comingMeta;
    StringPtr comingBody(new string);
    if (!storage->get_part(hid, *comingBody, comingMeta))
    {
        auto error_text = storage->get_last_error();

        throw std::runtime_error(
            string("libmulca get_part returns error: ") + error_text + " (" + stid + ":" + hid +
            ")");
    }
    setMetaIfNotEmpty(comingMeta);
    return comingBody;
}

ConstStringPtr MessageAccessLibmulca::getBody(const string& hid)
{
    if (hid == WHOLE_MESSAGE_HID)
    {
        return this->getWhole();
    }

    if (meta_.empty())
        if (setMeta()) return NULL;

    MetaParts::iterator mpi;
    string CONTENT_TYPE_TYPE("content_type.type");
    string TEXT("text"), LENGTH("length");
    mpi = meta_.find(hid);
    if (mpi != meta_.end() && !strcasecmp(mpi->second[CONTENT_TYPE_TYPE].c_str(), TEXT.c_str()))
    {
        if (headerText.fetch()) return NULL;

        setMetaIfNotEmpty(headerText.getMeta());

        int offs = 0;
        for (MetaParts::iterator i = meta_.begin(); i != mpi; i++)
            if (!strcasecmp(i->second[CONTENT_TYPE_TYPE].c_str(), TEXT.c_str()))
            {
                offs += boost::lexical_cast<int>(i->second[LENGTH]);
            }
        if (static_cast<size_t>(offs + boost::lexical_cast<int>(mpi->second[LENGTH].c_str())) >
            headerText.getText().size())
        {
            YLOG_G(error) << "MessageStorageMulca::getBody error: corrupted message body (mulcaid="
                          << stid << ", hid=" << hid << ")";
            return StringPtr(new string());
        }
        return StringPtr(new string(
            headerText.getText(), offs, boost::lexical_cast<int>(mpi->second[LENGTH].c_str())));
    }
    else if (mpi != meta_.end())
    {
        return getPart(hid);
    }
    else
    {
        return getBodyInlineMessage(hid);
    }
    // NOT REACHED
    return StringPtr();
}

ConstStringPtr MessageAccessLibmulca::getWhole()
{
    if (isCachedWhole_) return StringPtr(new string(whole_));

    StringPtr comingWhole(new string);
    string comingMeta;

    pa::async_stack_profiler prof(pa::mulca);
    prof.set_host(extractMulcaHost(stid));
    prof.set_request("get");

    if (!storage->get(*comingWhole, comingMeta))
    {
        auto error_text = storage->get_last_error();

        throw std::runtime_error(
            string("libmulca get returns error: ") + error_text + " (" + stid + ")");
    }
    setMetaIfNotEmpty(comingMeta);

    whole_ = *comingWhole;
    isCachedWhole_ = true;

    return comingWhole;
}

size_t MessageAccessLibmulca::getRfcSize() const
{
    return headerText.getRfcSize();
}

string MessageAccessLibmulca::getErrorText()
{
    return storage->get_last_error();
}

MetaAttributesPtr MessageAccessLibmulca::getMessageHeaderParsed(const string& hid)
{
    const ConstStringPtr hdr = getHeader(hid);
    if (!hdr) return MetaAttributesPtr();

    MetaAttributesPtr res(new MetaAttributes);
    HeaderParser headerParser(*res, *hdr);
    headerParser.parse(0, 0, hdr->size());

    return res;
}

int MessageAccessLibmulca::setMetaIfNotEmpty(const string& metaStr)
{
    if (!meta_.empty()) return 1;

    // fill this->meta using metaStr
    MetaParser mp;
    meta_.clear();
    int rc = mp.process(metaStr, &meta_, &inlm_, inlineLevel_ == 0);
    if (rc)
    {
        std::ostringstream s;
        s << __PRETTY_FUNCTION__ << ": "
          << "can't parse meta for " << stid << ": " << mp.parserError();
        throw std::runtime_error(s.str());
    }
    return rc;
}

int MessageAccessLibmulca::setMeta()
{
    int errorCode = 0;
    if (meta_.empty())
    {
        errorCode = headerText.fetch();
        if (errorCode == 0)
        {
            setMetaIfNotEmpty(headerText.getMeta());
        }
    }
    return errorCode;
}

int MessageAccessLibmulca::setMetaFromHeaderOnly()
{
    int errorCode = headerText.fetchHeader();
    if (errorCode == 0)
    {
        setMetaIfNotEmpty(headerText.getMeta());
    }
    return errorCode;
}

MessageStorageStringPtr MessageAccessLibmulca::findInlineMessage(const string& hid, int* sawoff)
{
    for (InlineMessages::iterator imi = inlm_.begin(); imi != inlm_.end(); imi++)
    {
        if (imi->first.size() < hid.size() &&
            !strncmp(imi->first.c_str(), hid.c_str(), imi->first.size()) &&
            '.' == hid[imi->first.size()])
        {
            if (!imi->second)
            {
                ConstStringPtr imes = getInlineMessage(imi->first);
                if (!imes) return MessageStorageStringPtr();
                imi->second = std::make_shared<MessageStorageString>(
                    imes, imi->first + ".", inlineLevel_ - 1);
            }
            if (sawoff) *sawoff = static_cast<int>(imi->first.size() + 1);
            return imi->second;
        }
    }
    return MessageStorageStringPtr();
}

MetaAttributesPtr MessageAccessLibmulca::getHeaderStructInlineMessage(const string& hid)
{
    int sawoff = 0;
    MessageStorageStringPtr inlmes = findInlineMessage(hid, &sawoff);
    if (!inlmes)
    {
        std::ostringstream s;
        s << __PRETTY_FUNCTION__ << "Can't find inline message. hid=" << hid;
        throw std::runtime_error(s.str());
    }
    return inlmes->getHeaderStruct(hid.substr(sawoff));
}

MetaLevelPtr MessageAccessLibmulca::getBodyStructInlineMessage(const string& hid)
{
    int sawoff = 0;
    MessageStorageStringPtr inlmes = findInlineMessage(hid, &sawoff);
    if (!inlmes) return MetaLevelPtr();
    return inlmes->getBodyStruct(hid.substr(sawoff));
}

ConstStringPtr MessageAccessLibmulca::getHeaderInlineMessage(const string& hid)
{
    int sawoff = 0;
    MessageStorageStringPtr inlmes = findInlineMessage(hid, &sawoff);
    if (!inlmes) return ConstStringPtr();
    return inlmes->getHeader(hid.substr(sawoff));
}

ConstStringPtr MessageAccessLibmulca::getBodyInlineMessage(const string& hid)
{
    int sawoff = 0;
    MessageStorageStringPtr inlmes = findInlineMessage(hid, &sawoff);
    if (!inlmes) return ConstStringPtr();
    return inlmes->getBody(hid.substr(sawoff));
}

ConstStringPtr MessageAccessLibmulca::getInlineMessage(const string& hid)
{
    ConstStringPtr message = getBody(hid);
    MetaParts::const_iterator it = meta_.find(hid);
    if (it == meta_.end())
    {
        return message;
    }
    const MetaAttributes& attributes = it->second;
    MetaAttributes::const_iterator attrIt = attributes.find("content_transfer_encoding");
    if (attrIt == attributes.end())
    {
        return message;
    }
    string transferEncoding = attrIt->second;
    if (boost::algorithm::iequals(transferEncoding, "base64"))
    {
        // *message = decode_base64(*message);
        message.reset(new string(decode_base64(*message)));
    }
    else if (boost::algorithm::iequals(transferEncoding, "quoted-printable"))
    {
        message.reset(new string(decode_qp(*message)));
        // *message = decode_qp(*message);
    }
    return message;
}

MessageAccessLibmulca::HeaderTextCacher::HeaderTextCacher(Storage& storage, Storage& httpStorage)
    : storage(storage), httpStorage(httpStorage)
{
}

int MessageAccessLibmulca::HeaderTextCacher::fetch()
{
    if (!isCached())
    {
        pa::async_stack_profiler prof(pa::mulca);
        prof.set_host(extractMulcaHost(storage.getStid()));
        prof.set_request("get_header_text");

        if (!httpStorage.get_header_text(header_, text_, meta_))
        {
            auto error_text = storage.get_last_error();
            throw std::runtime_error(
                string("libmulca get_header_text returns error: ") + error_text + " (" +
                storage.getStid() + ")");
        }
        rfcSize = httpStorage.get_rfc_size();
        cached = true;
        headerCached_ = true;
    }

    return 0;
}

int MessageAccessLibmulca::HeaderTextCacher::fetchHeader()
{
    if (!isHeaderCached())
    {
        pa::async_stack_profiler prof(pa::mulca);
        prof.set_host(extractMulcaHost(storage.getStid()));
        prof.set_request("get_header");

        if (!httpStorage.get_header(header_, meta_))
        {
            auto error_text = storage.get_last_error();
            throw std::runtime_error(
                string("libmulca get_header returns error: ") + error_text + " (" +
                storage.getStid() + ")");
        }
        rfcSize = httpStorage.get_rfc_size();
        headerCached_ = true;
    }

    return 0;
}

const string& MessageAccessLibmulca::HeaderTextCacher::getMeta() const
{
    return meta_;
}

const string& MessageAccessLibmulca::HeaderTextCacher::getHeader() const
{
    return header_;
}

const string& MessageAccessLibmulca::HeaderTextCacher::getText() const
{
    return text_;
}

size_t MessageAccessLibmulca::HeaderTextCacher::getRfcSize() const
{
    return rfcSize;
}

bool MessageAccessLibmulca::HeaderTextCacher::isHeaderCached() const
{
    return headerCached_;
}

bool MessageAccessLibmulca::HeaderTextCacher::isCached() const
{
    return cached;
}

} // namespace mbody
} // namespace yimap
