#include "message_loader.h"
#include "message_access.h"

#include <yplatform/find.h>
#include <yplatform/encoding/iterators/crlf.h>
#include <yplatform/log.h>
#include <yplatform/util/sstream.h>
#include <boost/lexical_cast.hpp>
#include <boost/range/iterator.hpp>

#include <vector>

namespace yimap { namespace mbody {

namespace ph = std::placeholders;

class GetHid : public std::enable_shared_from_this<GetHid>
{
public:
    using Handler = std::function<void(std::string err, std::string hid)>;
    using HeaderStruct = MessageStorage::HeaderStruct;

    GetHid(MessageStoragePtr messageStorage, std::string imapHid, Handler handler)
        : messageStorage(messageStorage)
        , imapHid(std::move(imapHid))
        , handler(std::move(handler))
        , hidStream(hid, this->imapHid.length() + 10)
    {
    }

    void operator()()
    {
        processNextPart();
    }

private:
    void processNextPart()
    {
        if (afterTheLastDot < imapHid.length())
        {
            auto nextDotPos = imapHid.find('.', afterTheLastDot);
            if (nextDotPos == std::string::npos)
            {
                nextDotPos = imapHid.length();
            }
            hidStream << "." << imapHid.substr(afterTheLastDot, nextDotPos - afterTheLastDot);
            auto cb = std::bind(&GetHid::headerStructCallback, shared_from_this(), ph::_1, ph::_2);
            afterTheLastDot = nextDotPos + 1;
            messageStorage->getHeaderStruct(hid, std::move(cb));
        }
        else
        {
            handler("", std::move(hid));
        }
    }

    void headerStructCallback(std::string err, HeaderStruct headerStruct)
    {
        if (!err.empty())
        {
            handler(std::move(err), "");
            return;
        }
        if (headerStruct.contentType == "message" && headerStruct.contentSubtype == "rfc822")
        {
            hidStream << ".1";
        }
        processNextPart();
    }

    MessageStoragePtr messageStorage;
    std::string imapHid;
    Handler handler;
    std::string hid = "1";
    std::size_t afterTheLastDot = 0;
    yplatform::sstream hidStream; // stream to this->hid
};

MessageLoader::MessageLoader(
    const std::string& stid,
    const MbodyStorageOptions& storageOptions,
    ContextPtr context,
    std::function<void(const std::string&)> logWarning,
    const macs::MimeParts& mimeParts,
    time_t receiveDate)
    : stid(stid)
    , context(context)
    , options(storageOptions)
    , logWarning(std::move(logWarning))
    , haveMimeParts(!mimeParts.empty())
    , receiveDate(receiveDate)
{
    auto storage = createStorage(stid, options, context);
    messageAccess = std::make_shared<MessageAccessLibmulca>(storage);

    auto factory = yplatform::find<MessageStorageFactory>("message_storage");
    messageStorage = factory->create(context, stid, mimeParts);
}

yplatform::future::future<void> MessageLoader::load(
    LoadMessageMode mode,
    const std::string& part,
    size_t expectedSize,
    MessageDataHandlerPtr handler)
{
    yplatform::future::promise<void> completePromise;

    bool withHeader = mode == LoadMessage || mode == LoadMessagePart || mode == LoadMessageHeader ||
        mode == LoadMessageHeaderPart;

    bool withBody = mode == LoadMessage || mode == LoadMessagePart || mode == LoadMessageBody ||
        mode == LoadMessageBodyPart;

    auto backendType = options.getBackendType(context->uniq_id(), haveMimeParts, receiveDate);
    Future future;
    if (backendType == MbodyStorageOptions::BackendType::Mulca)
    {
        future = loadFromMulca(part, withHeader, withBody);
    }
    else if (backendType == MbodyStorageOptions::BackendType::MDS)
    {
        future = loadFromMDS(part, withHeader, withBody);
    }
    else
    {
        future = loadFromBoth(
            part, withHeader, withBody, backendType == MbodyStorageOptions::BackendType::BothMDS);
    }

    auto self = shared_from_this();
    future.add_callback([this, self, future, expectedSize, handler, completePromise]() mutable {
        Result res;
        try
        {
            res = future.get();
        }
        catch (const std::exception& e)
        {
            res.err = std::string("MessageLoader::load error in handler: ") + e.what();
        }
        if (!res.err.empty())
        {
            handler->onMessageError(std::move(res.err));
            completePromise.set();
            return;
        }
        if (res.rfcSize && res.rfcSize != expectedSize)
        {
            std::stringstream ss;
            ss << "message loader error, wrong db size: size_from_db=" << expectedSize
               << ", real_message_size=" << res.rfcSize << ", stid=" << stid;
            handler->onLogWarning(ss.str());
        }
        handler->onMessage(std::make_shared<std::string>(std::move(res.msg)), res.rfcSize);
        completePromise.set();
    });

    return completePromise;
}

MessageLoader::Future MessageLoader::loadFromBoth(
    const std::string& part,
    bool withHeader,
    bool withBody,
    bool returnMDS)
{
    auto mdsFuture = loadFromMDS(part, withHeader, withBody);
    auto mulcaFuture = loadFromMulca(part, withHeader, withBody);

    try
    {
        auto mulcaResult = mulcaFuture.get();
        auto mdsResult = mdsFuture.get();

        if (mulcaResult.msg != mdsResult.msg)
        {
            std::stringstream str;
            str << "MessageLoader wrong mdsResult:"
                << " mulca[" << removeLineEndingsAndTabs(mulcaResult.msg) << "]"
                << " err[" << mulcaResult.err << "]"
                << " != mds[" << removeLineEndingsAndTabs(mdsResult.msg) << "]"
                << " err[" << mdsResult.err << "]"
                << " return " << (returnMDS ? "mds" : "mulca") << " stid=" << stid;
            logWarning(str.str());
        }
    }
    catch (const std::exception& e)
    {
        logWarning(std::string("MessageLoader::loadFromBoth error: ") + e.what());
    }

    return returnMDS ? mdsFuture : mulcaFuture;
}

MessageLoader::Future MessageLoader::loadFromMDS(
    const std::string& part,
    bool withHeader,
    bool withBody)
{
    Promise promise;
    auto cb = std::bind(
        &MessageLoader::doLoadFromMDS,
        shared_from_this(),
        promise,
        part,
        withHeader,
        withBody,
        ph::_1,
        ph::_2);
    auto getHid = std::make_shared<GetHid>(messageStorage, part, std::move(cb));
    (*getHid)();
    return promise;
}

void MessageLoader::doLoadFromMDS(
    Promise promise,
    const std::string& part,
    bool withHeader,
    bool withBody,
    std::string err,
    std::string hid)
{
    // Some client just can't get that there are some emails without 1.1 mime part
    if (!err.empty() && part == "1")
    {
        doLoadFromMDS(promise, "", withHeader, withBody, "", "1");
        return;
    }
    if (!err.empty())
    {
        Result result = { "", 0, std::move(err) };
        promise.set(std::move(result));
        return;
    }
    auto self = shared_from_this();
    auto cb = [this, self, promise](std::string err, std::string msg) mutable {
        if (!err.empty())
        {
            Result result = { "", 0, std::move(err) };
            promise.set(std::move(result));
            return;
        }
        auto getSizeHandler =
            [this, self, promise, msg = std::move(msg)](std::string err, std::size_t size) mutable {
                Result result = { std::move(msg), 0, "" };
                if (err.empty())
                {
                    result.rfcSize = size;
                }
                promise.set(std::move(result));
            };
        messageStorage->getMessageSize(std::move(getSizeHandler));
    };
    if (withHeader && withBody)
    {
        messageStorage->getWhole(std::move(hid), std::move(cb));
    }
    else if (withHeader)
    {
        messageStorage->getHeaders(std::move(hid), std::move(cb));
    }
    else if (withBody)
    {
        messageStorage->getBody(std::move(hid), std::move(cb));
    }
}

MessageLoader::Future MessageLoader::loadFromMulca(std::string part, bool withHeader, bool withBody)
{
    auto hid = prepareHid(part);
    ConstStringPtr header;
    ConstStringPtr body;
    size_t rfcSize = 0;

    // Try load with given hid.
    bool error = loadByHid(hid, withHeader, withBody, header, body, rfcSize);

    // Some client on Nokia mobile telephone just can't get that there are some emails without 1.1
    // mime part
    if (error && part == "1")
    {
        hid = "1";
        part = "";
        error = loadByHid(hid, withHeader, withBody, header, body, rfcSize);
    }

    if (error)
    {
        auto err = "failed to load message, stid=" + stid + ", part=" + part +
            ", message_access_error='" + messageAccess->getErrorText() + "', exception='" +
            loader_error + "'";
        Result result = { "", 0, std::move(err) };
        Promise promise;
        promise.set(result);
        return promise;
    }

    std::string msg;
    yplatform::sstream str(msg, (header ? header->length() + 2 : 0) + (body ? body->length() : 0));
    if (header)
    {
        str << *header;
        if (part.empty())
        {
            str << "\r\n";
        }
    }
    if (body)
    {
        str << *body;
    }

    Result result = { std::move(msg), rfcSize, "" };
    Promise promise;
    promise.set(result);
    return promise;
}

bool MessageLoader::loadByHid(
    const std::string& hid,
    bool withHeader,
    bool withBody,
    ConstStringPtr& header,
    ConstStringPtr& body,
    size_t& rfcSize)
{
    bool error = false;

    try
    {
        if (!error && withBody)
        {
            body = messageAccess->getBody(hid);
            error = !body.get();
        }
        if (!error && withHeader)
        {
            header = messageAccess->getHeader(hid);
            error = !header.get();
        }
    }
    catch (const std::exception& e)
    {
        loader_error = e.what();
        error = true;
    }
    catch (...)
    {
        loader_error = "unknown error";
        error = true;
    }

    if (!error)
    {
        rfcSize = messageAccess->getRfcSize();
    }

    return error;
}

enum MimePartType
{
    ERROR,
    RFC822_MIME_LETTER,
    NOT_RFC822_LETTER,
};

int MessageLoader::getPartType(const std::string& hid)
{
    const char* const CONTENT_TYPE = "content_type.type";
    const char* const CONTENT_SUBTYPE = "content_type.subtype";
    const char* const MESSAGE_TYPE = "message";
    const char* const MESSAGE_SUBTYPE = "rfc822";

    std::shared_ptr<MetaAttributes> parsedHeader;

    try
    {
        parsedHeader = messageAccess->getHeaderStruct(hid);
    }
    catch (...)
    {
    }

    if (!parsedHeader) return ERROR;

    if (((*parsedHeader)[CONTENT_TYPE] == MESSAGE_TYPE) &&
        ((*parsedHeader)[CONTENT_SUBTYPE] == MESSAGE_SUBTYPE))
        return RFC822_MIME_LETTER;
    else
        return NOT_RFC822_LETTER;
}

string MessageLoader::prepareHid(const std::string& imapHid)
{
    std::vector<int> parsedHid;
    size_t afterTheLastDot = 0;
    while (afterTheLastDot < imapHid.length())
    {
        size_t nextDotPos = imapHid.find('.', afterTheLastDot);
        if (nextDotPos == std::string::npos)
        {
            nextDotPos = imapHid.length();
        }
        parsedHid.push_back(boost::lexical_cast<int>(
            imapHid.substr(afterTheLastDot, nextDotPos - afterTheLastDot)));
        afterTheLastDot = nextDotPos + 1;
    }

    std::string hid = "1";

    size_t index = 0;
    while (index < parsedHid.size())
    {
        hid += ".";
        hid += boost::lexical_cast<string>(parsedHid[index]);

        int code = getPartType(hid);
        if (code == ERROR)
        {
            break;
        }

        if (code == RFC822_MIME_LETTER) hid += ".1";

        ++index;
    }
    return hid;
}

} // namespace mbody
} // namespace yimap
