#pragma once

#include <backend/mbody/types.h>
#include <backend/backend_types.h>
#include <common/context.h>
#include <backend/mbody/storage/message_storage.h>

namespace yimap { namespace mbody {

enum LoadMessageMode
{
    LoadMessage,
    LoadMessagePart,
    LoadMessageBody,
    LoadMessageBodyPart,
    LoadMessageHeader,
    LoadMessageHeaderPart,
};

class MessageDataHandler
{
public:
    virtual ~MessageDataHandler()
    {
    }
    virtual void onMessage(StringPtr, size_t rfc822Size = 0) = 0;
    virtual void onMessageError(const std::string&) = 0;
    virtual void onLog(const std::string& text) = 0;
    virtual void onLogWarning(const std::string& text) = 0;
};

typedef std::shared_ptr<MessageDataHandler> MessageDataHandlerPtr;
class MessageAccessLibmulca;

class MessageLoader : public std::enable_shared_from_this<MessageLoader>
{
public:
    MessageLoader(
        const std::string& stid,
        const MbodyStorageOptions&,
        ContextPtr context,
        std::function<void(const std::string&)> logWarning,
        const macs::MimeParts& mimeParts,
        time_t receiveDate);

    yplatform::future::future<void> load(
        LoadMessageMode mode,
        const std::string& part,
        size_t expectedSize,
        MessageDataHandlerPtr handler);

protected:
    struct Result
    {
        std::string msg;
        size_t rfcSize;
        std::string err;
    };

    using Future = yplatform::future::future<Result>;
    using Promise = yplatform::future::promise<Result>;

    Future loadFromMDS(const std::string& part, bool withHeader, bool withBody);
    Future loadFromMulca(std::string part, bool withHeader, bool withBody);
    Future loadFromBoth(const std::string& part, bool withHeader, bool withBody, bool returnMDS);

    void doLoadFromMDS(
        Promise promise,
        const std::string& part,
        bool withHeader,
        bool withBody,
        std::string err,
        std::string hid);

    std::string removeLineEndingsAndTabs(const std::string& s)
    {
        std::string result;
        for (const auto c : s)
        {
            if (c == '\n')
            {
                result += "\\n";
            }
            else if (c == '\r')
            {
                result += "\\r";
            }
            else if (c == '\t')
            {
                result += "\\t";
            }
            else
            {
                result += c;
            }
        }
        return result;
    }

    bool loadByHid(
        const std::string& hid,
        bool withHeader,
        bool withBody,
        ConstStringPtr& header,
        ConstStringPtr& body,
        size_t& rfcSize);

    int getPartType(const std::string& hid);
    std::string prepareHid(const std::string& imapHid);

protected:
    std::string stid;
    ContextPtr context;
    std::shared_ptr<MessageAccessLibmulca> messageAccess;
    MessageDataHandlerPtr handler;
    MbodyStorageOptions options;
    std::function<void(const std::string&)> logWarning;
    MessageStoragePtr messageStorage;
    bool haveMimeParts = false;
    time_t receiveDate = 0;

    std::string loader_error;
};

} // namespace mbody
} // namespace yimap
