#include "mbody_stream_parser.h"

#include <boost/property_tree/xml_parser.hpp>
#include <yplatform/ptree.h>

#include <stdexcept>
#include <cassert>
#include <sstream>

namespace yimap { namespace mbody {

//-----------------------------------------------------------------------------
// Support class required to find both LF and CRLF line ending.

MbodyStreamParser::CRLF MbodyStreamParser::CRLF::trySkip(const std::string& data, size_t offset)
{
    auto tail = data.substr(offset, 2);
    if (tail.empty() || tail == "\r") return CRLF{ 0, true };

    if (tail[0] == '\n') return CRLF{ 1u, false };

    if (tail == "\r\n") return CRLF{ 2u, false };

    return CRLF{ 0, false };
}

static const std::string CLOSE_META_TAG = "</message>";
static const std::string CLOSE_HEADER_LF = "\n";
static const std::string EMPTY_DELIMITER = "";

//-----------------------------------------------------------------------------

MbodyStreamParser::MbodyStreamParser(SkipPattern skipPattern)
    : separators({ CLOSE_META_TAG, CLOSE_HEADER_LF, EMPTY_DELIMITER })
{
    if (skipPattern == SkipPattern::Normal)
    {
        shouldSkip = { false, true, false };
    }
    else
    {
        shouldSkip = { false, false, false };
    }
}

void MbodyStreamParser::parse(const char* chunk, size_t length)
{
    data.append(chunk, length);

    while (!separators.empty())
    {
        const auto& delimiter = separators.front();
        if (delimiter.empty())
        {
            offset = data.size();
            return;
        }

        auto pos = data.find(delimiter, offset);
        if (pos == std::string::npos)
        {
            offset = std::max(
                offset, data.size() > delimiter.size() ? data.size() - delimiter.size() : 0);
            return;
        }
        auto crlf = CRLF::trySkip(data, pos + delimiter.size());
        if (crlf.needData)
        {
            offset = pos;
            return;
        }
        offset = pos + delimiter.size();
        if (crlf.size)
        {
            takeChunk(crlf.size);
        }
    }
}

void MbodyStreamParser::parseRfcSize(std::string& xml)
{
    try
    {
        std::stringstream ss(xml);
        yplatform::ptree resp;
        boost::property_tree::read_xml(ss, resp);
        auto message = resp.get_child_optional("message");

        if (message)
        {
            for (auto& part : *message)
            {
                size_t id = part.second.get("<xmlattr>.id", 0);
                if (id == 1)
                {
                    size_t offset = part.second.get("<xmlattr>.offset", 0);
                    size_t length = part.second.get("<xmlattr>.length", 0);
                    rfcSize = offset + length;
                    break;
                }
            }
        }
    }
    catch (...)
    {
    }
}

bool MbodyStreamParser::takeChunk(size_t crlfSize)
{
    if (separators.empty())
    {
        return false;
    }

    assert(shouldSkip.size() == separators.size());
    auto chunkSize = offset + (shouldSkip.front() ? 0 : crlfSize);
    chunks.emplace_back(data.substr(0, chunkSize));
    data.erase(0, offset + crlfSize);

    separators.pop_front();
    shouldSkip.pop_front();
    offset = 0;

    return true;
}

MbodyData MbodyStreamParser::takeData()
{
    if (!takeChunk())
        return MbodyData(
            std::make_exception_ptr(std::runtime_error("Invalid MbodyData structure")));

    chunks.resize(3, "");

    parseRfcSize(chunks[0]);

    MbodyData data;
    data.meta = std::move(chunks[0]);
    data.header = std::move(chunks[1]);
    data.text = std::move(chunks[2]);
    data.rfcSize = rfcSize;

    return data;
}

//-----------------------------------------------------------------------------
MbodyData::MbodyData(std::exception_ptr err) : error(err)
{
}

bool MbodyData::takeMetaBody(std::string& meta, std::string& body)
{
    if (error == nullptr)
    {
        meta = std::move(this->meta);
        body = std::move(this->header);
        body += std::move(this->text);
        return true;
    }
    else
    {
        return false;
    }
}
bool MbodyData::takeMetaHeader(std::string& meta, std::string& header)
{
    if (error == nullptr)
    {
        meta = std::move(this->meta);
        header = std::move(this->header);
        return true;
    }
    else
    {
        return false;
    }
}
bool MbodyData::takeMetaHeaderText(std::string& meta, std::string& header, std::string& text)
{
    if (error == nullptr)
    {
        meta = std::move(this->meta);
        header = std::move(this->header);
        text = std::move(this->text);
        return true;
    }
    else
    {
        return false;
    }
}

} // namespace mbody
} // namespace yimap
