#pragma once

#include <boost/optional.hpp>
#include <yplatform/util/sstream.h>
#include <macs/types.h>
#include <backend/mbody/storage/raw_storage.h>
#include <backend/mbody/util/meta_parser.h>

namespace yimap { namespace mbody {

namespace ph = std::placeholders;

static const std::string XML_PREFIX = "<?xml";
static const std::string XML_SUFFIX = "</message>";

class MimeStorage : public std::enable_shared_from_this<MimeStorage>
{
public:
    using HeaderStruct = MessageStorage::HeaderStruct;

    struct MetaData
    {
        std::size_t xmlLength;
        std::size_t messageLength;
        ConstMetaPartsPtr metaParts;
    };

    using Handler = std::function<void(std::string err, MetaData)>;

    MimeStorage(RawStoragePtr storage, const macs::MimeParts& mimeParts) : storage(storage)
    {
        if (!mimeParts.empty())
        {
            metaData = convert(mimeParts);
        }
    }

    void get(Handler handler)
    {
        auto ret = meta();
        if (ret)
        {
            handler("", std::move(*ret));
        }
        else
        {
            getMetaFromXml(std::move(handler));
        }
    }

private:
    boost::optional<MetaData> meta()
    {
        std::lock_guard<std::mutex> lock(mutex);
        return metaData;
    }

    void setMeta(MetaData data)
    {
        std::lock_guard<std::mutex> lock(mutex);
        metaData = data;
    }

    void getMetaFromXml(Handler handler)
    {
        storage->getMimeStructure(std::bind(
            &MimeStorage::handleXml, shared_from_this(), std::move(handler), ph::_1, ph::_2));
    }

    void handleXml(const Handler& handler, std::string err, std::string xml)
    {
        if (!err.empty())
        {
            handler("MimeStorage can't get xml: " + err, MetaData());
            return;
        }

        MetaPartsPtr parts = std::make_shared<MetaParts>();
        InlineMessages inlm;
        MetaParser mp;
        const int rc = mp.process(xml, parts.get(), &inlm, false);
        if (rc)
        {
            handler("MimeStorage can't parse xml: " + mp.parserError(), MetaData());
            return;
        }

        MetaData res;
        res.xmlLength = xml.length();
        res.messageLength = extractMessageLength(*parts);
        res.metaParts = parts;

        setMeta(res);
        handler("", std::move(res));
    }

    std::size_t extractMessageLength(const MetaParts& metaParts)
    {
        auto mpi = metaParts.find("1");
        if (mpi == metaParts.end())
        {
            return 0;
        }
        try
        {
            auto bodyOffset = std::stoull(mpi->second.at("offset"));
            auto bodyLength = std::stoull(mpi->second.at("length"));
            return bodyOffset + bodyLength;
        }
        catch (...)
        {
            return 0;
        }
    }

    MetaData convert(const macs::MimeParts& mimeParts)
    {
        MetaParts parts;
        for (const auto& part : mimeParts)
        {
            MetaAttributes attributes;
            attributes["offset"] = std::to_string(part.offsetBegin());
            attributes["length"] = std::to_string(part.length());
            attributes["content_type.type"] = part.contentType();
            attributes["content_type.subtype"] = part.contentSubtype();
            attributes["content_type.name"] = part.name();
            attributes["content_type.charset"] = part.charset();
            attributes["content_transfer_encoding"] = part.encoding();
            attributes["content_disposition.value"] = part.contentDisposition();
            attributes["content_disposition.filename"] = part.fileName();
            attributes["content_id"] = part.cid();
            parts.emplace(part.hid(), std::move(attributes));
        }
        MetaData result;
        // when MetaData taken from macs::MimeParts all offsets are incremented on xmlLength
        // so in this case we assume that xmlLength and messageLength equals 0
        result.xmlLength = 0;
        result.messageLength = 0;
        result.metaParts = std::make_shared<const MetaParts>(std::move(parts));
        return result;
    }

    RawStoragePtr storage;
    std::mutex mutex;
    boost::optional<MetaData> metaData;
};

using MimeStoragePtr = std::shared_ptr<MimeStorage>;

} // namespace mbody
} // namespace yimap
