#include "mulcagate_storage.h"
#include "mbody_stream_parser.h"

#include <backend/backend_types.h>
#include <ymod_httpclient/call.h>
#include <yplatform/find.h>
#include <boost/algorithm/string.hpp>

namespace yimap { namespace mbody {

class MbodyHandler : public ymod_http_client::response_handler
{
public:
    MbodyHandler(
        MbodyStreamParser::SkipPattern skipPattern = MbodyStreamParser::SkipPattern::Normal)
        : parser(skipPattern)
    {
    }

    static boost::shared_ptr<MbodyHandler> create(
        MbodyStreamParser::SkipPattern skipPattern = MbodyStreamParser::SkipPattern::Normal)
    {
        return boost::make_shared<MbodyHandler>(skipPattern);
    }

    void handle_data(const char* chunk, unsigned long long size) override
    {
        parser.parse(chunk, size);
    }

    void handle_data_end() override
    {
    }

    MbodyData takeData()
    {
        return parser.takeData();
    }

protected:
    MbodyStreamParser parser;
};

using MbodyHandlerPtr = boost::shared_ptr<MbodyHandler>;

namespace rc {

using RateControllerPtr = ymod_ratecontroller::rate_controller_ptr;
using HttpClientPtr = boost::shared_ptr<yhttp::call>;
using RemoteInfoPtr = ymod_httpclient::remote_point_info_ptr;

class GetUrlOp : public std::enable_shared_from_this<GetUrlOp>
{
    using PromiseVoid = yplatform::future::promise<void>;
    using FutureVoid = yplatform::future::future<void>;
    using RcCompletionHandler = ymod_ratecontroller::completion_handler;

public:
    GetUrlOp(
        RateControllerPtr rateController,
        HttpClientPtr client,
        ContextPtr context,
        MbodyHandlerPtr handler,
        RemoteInfoPtr rmInfo,
        const std::string& url)
        : client(client)
        , context(context)
        , handler(handler)
        , rmInfo(rmInfo)
        , url(url)
        , rateController(rateController)
    {
    }

    FutureVoid run()
    {
        PromiseVoid prom;

        auto self = shared_from_this();
        rateController->post([prom, self, this](
                                 boost::system::error_code ec,
                                 RcCompletionHandler rcHandler) mutable {
            if (ec)
            {
                prom.set_exception(std::runtime_error("rate controller error: " + ec.message()));
                return rcHandler();
            }

            auto future = client->get_url(context, handler, rmInfo, url);
            future.add_callback([prom, future, rcHandler]() mutable {
                rcHandler();

                try
                {
                    future.get();
                    prom.set();
                }
                catch (...)
                {
                    prom.set_exception(std::current_exception());
                }
            });
        });

        return prom;
    }

private:
    HttpClientPtr client;
    ContextPtr context;
    MbodyHandlerPtr handler;
    RemoteInfoPtr rmInfo;
    std::string url;

    RateControllerPtr rateController;
};

void getUrl(
    RateControllerPtr rateController,
    HttpClientPtr client,
    ContextPtr context,
    MbodyHandlerPtr handler,
    RemoteInfoPtr rmInfo,
    const std::string& url)
{
    auto op = std::make_shared<GetUrlOp>(rateController, client, context, handler, rmInfo, url);
    op->run().get();
}

}

MulcagateStorage::MulcagateStorage(
    RateControllerPtr rateController,
    const std::string& stid,
    const MbodyStorageOptions& storageOptions,
    ContextPtr context)
    : stid(stid), storageOptions(storageOptions), context(context), rateController(rateController)
{
}

MbodyData MulcagateStorage::loadByHttp(const std::string& part)
{
    auto httpService = yplatform::find<yhttp::call>("http_client");
    ymod_httpclient::timeouts timeouts;
    timeouts.total = storageOptions.mulcagateTimeout;
    auto remoteInfo = httpService->make_rm_info(storageOptions.mulcagateHost, timeouts);
    try
    {
        if (part.empty())
        {
            auto handler = MbodyHandler::create(MbodyStreamParser::SkipPattern::Noskip);
            rc::getUrl(rateController, httpService, context, handler, remoteInfo, makeUrl());
            return handler->takeData();
        }
        else if (part == "meta")
        {
            auto handler = MbodyHandler::create();
            rc::getUrl(rateController, httpService, context, handler, remoteInfo, makeUrl("xml"));
            rc::getUrl(rateController, httpService, context, handler, remoteInfo, makeUrl("meta"));
            return handler->takeData();
        }
        else if (part == "header_text")
        {
            auto handler = MbodyHandler::create();
            rc::getUrl(rateController, httpService, context, handler, remoteInfo, makeUrl("xml"));
            rc::getUrl(rateController, httpService, context, handler, remoteInfo, makeUrl("meta"));
            rc::getUrl(rateController, httpService, context, handler, remoteInfo, makeUrl("text"));
            return handler->takeData();
        }
        else
        {
            auto handler = MbodyHandler::create();
            rc::getUrl(rateController, httpService, context, handler, remoteInfo, makeUrl("xml"));
            handler->handle_data("\n\n", 2);
            rc::getUrl(
                rateController, httpService, context, handler, remoteInfo, makeUrl("part", part));
            return handler->takeData();
        }
    }
    catch (const std::exception& e)
    {
        lastError = e.what();
        return MbodyData(std::current_exception());
    }
}

std::string MulcagateStorage::makeUrl(const std::string& type, const std::string& part) const
{
    std::vector<std::string> attrs;
    if (!storageOptions.mulcagateAttrs.empty()) attrs.push_back(storageOptions.mulcagateAttrs);
    if (!type.empty()) attrs.emplace_back("gettype=" + type);
    if (!part.empty()) attrs.emplace_back("part=" + part);

    std::string url = "gate/get/";
    url += stid;
    if (!attrs.empty())
    {
        url += "?";
        url += boost::join(attrs, "&");
    }
    return url;
}

bool MulcagateStorage::get(std::string& msg, std::string& meta)
{
    auto result = loadByHttp();
    rfcSize = result.getRfcSize();
    return result.takeMetaBody(meta, msg);
}

bool MulcagateStorage::get_header(std::string& header, std::string& meta)
{
    // auto result = loadByHttp();
    auto result = loadByHttp("meta");
    rfcSize = result.getRfcSize();
    return result.takeMetaHeader(meta, header);
}

bool MulcagateStorage::get_header_text(std::string& header, std::string& text, std::string& meta)
{
    auto result = loadByHttp("header_text");
    rfcSize = result.getRfcSize();
    return result.takeMetaHeaderText(meta, header, text);
}

bool MulcagateStorage::get_part(const std::string& part_id, std::string& msg, std::string& meta)
{
    std::string fakeHeader;
    auto result = loadByHttp(part_id);
    rfcSize = result.getRfcSize();
    return result.takeMetaHeaderText(meta, fakeHeader, msg);
}

size_t MulcagateStorage::get_rfc_size() const
{
    return rfcSize;
}

std::string MulcagateStorage::get_last_error() const
{
    return lastError;
}

} // namespace mbody
} // namespace yimap
