#include "raw_storage.h"
#include <common/types.h>
#include <boost/algorithm/string.hpp>
#include <ymod_httpclient/cluster_client.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/find.h>
#include <yplatform/util/sstream.h>
#include <yplatform/time_traits.h>

namespace yimap { namespace mbody {

namespace ph = std::placeholders;

class RawStorageImpl
    : public RawStorage
    , public std::enable_shared_from_this<RawStorageImpl>
{
    using RateControllerPtr = ymod_ratecontroller::rate_controller_ptr;
    using RcCompletionHandler = ymod_ratecontroller::completion_handler;

public:
    RawStorageImpl(
        const StorageOptionsConstPtr& options,
        const ContextPtr& context,
        const std::string& stid,
        const RateControllerPtr& rateController)
        : options(options)
        , warningLog(YGLOBAL_LOG_SERVICE, "warning_log")
        , context(context)
        , stid(truncateStid(stid, options->stidTruncatePrefix))
        , rateController(rateController)
    {
    }

    void getMimeStructure(Handler handler) const override
    {
        RequestData data;
        data.url = makeUrl(RequestType::XML);
        data.handler = std::move(handler);
        doRequest(std::move(data));
    }

    void get(Handler handler) const override
    {
        get(0, LengthOpt(), std::move(handler));
    }

    void get(const std::size_t offset, LengthOpt length, Handler handler) const override
    {
        if (length && *length == 0)
        {
            return handler(std::string(), std::string());
        }

        RequestData data;
        data.url = makeUrl(RequestType::BLOB);
        data.headers = makeHeaders(offset, length);
        data.handler = std::move(handler);
        doRequest(std::move(data));
    }

private:
    struct RequestData
    {
        Handler handler;
        std::string url;
        std::string headers;
        std::size_t retryCount = 0;
    };

    enum class RequestType
    {
        BLOB,
        XML
    };

    void doRequest(RequestData data) const
    {
        auto self = shared_from_this();
        rateController->post([data = std::move(data), self, this](
                                 ErrorCode ec, const RcCompletionHandler& rcHandler) {
            if (ec)
            {
                YLOG(warningLog, warning) << "Raw storage rc error: " << ec.message();
                rcHandler();
                data.handler(ec.message(), "");
                return;
            }

            auto request = yhttp::request::GET(data.url, data.headers);
            auto client = yplatform::find<yhttp::cluster_client>("storage_client");
            client->async_run(
                context,
                std::move(request),
                [data, rcHandler](ErrorCode ec, yhttp::response response) {
                    rcHandler();
                    if (ec)
                    {
                        data.handler(ec.message(), "");
                    }
                    else if (response.status / 100 != 2)
                    {
                        std::string msg;
                        yplatform::sstream str(
                            msg, response.reason.length() + sizeof("bad response code: xxx "));
                        str << "bad response code: " << response.status << " " << response.reason;
                        data.handler(std::move(msg), "");
                    }
                    else
                    {
                        data.handler("", std::move(response.body));
                    }
                });
        });
    }

    std::string makeUrl(const RequestType type) const
    {
        std::vector<std::string> attrs;
        if (type == RequestType::XML)
        {
            attrs.push_back("gettype=xml");
        }
        if (!options->extraAttrs.empty())
        {
            attrs.push_back(options->extraAttrs);
        }

        std::string url;
        auto hint =
            stid.length() + options->extraAttrs.length() + sizeof("/gate/get/?gettype=xml&&&&");
        yplatform::sstream str(url, hint);
        str << "/gate/get/" << stid;
        if (!attrs.empty())
        {
            str << "?" << boost::join(attrs, "&");
        }
        return url;
    }

    std::string makeHeaders(const std::size_t offset, LengthOpt length) const
    {
        std::string res;
        yplatform::sstream str(res, sizeof("Range: bytes=1000000500-2000000500\r\n"));

        str << "Range: bytes=" << offset << "-";
        if (length)
        {
            str << offset + *length - 1;
        }
        str << "\r\n";

        return res;
    }

    std::string truncateStid(const std::string& stid, const std::string& prefix) const
    {
        if (!prefix.empty() && stid.length() >= prefix.length() &&
            std::equal(prefix.begin(), prefix.end(), stid.begin()))
        {
            // erase prefix
            return stid.substr(prefix.length());
        }
        else
        {
            return stid;
        }
    }

    StorageOptionsConstPtr options;
    mutable yplatform::log::source warningLog;
    const ContextPtr context;
    const std::string stid;
    RateControllerPtr rateController;
};

RawStoragePtr createRawStorage(
    const StorageOptionsConstPtr& options,
    const ContextPtr& context,
    const std::string& stid)
{
    auto rcModule = yplatform::find<ymod_ratecontroller::rate_controller_module>("rate_controller");
    auto controller = rcModule->get_controller("mulcagate");
    return std::make_shared<RawStorageImpl>(options, context, stid, controller);
}

} // namespace mbody
} // namespace yimap
