#ifndef CHARACTER_CONVERTER_H_
#define CHARACTER_CONVERTER_H_

#include <butil/StrUtils/Iconv.h>
#include <butil/StrUtils/pystrutils.h>
#include <string>

namespace yimap {

/**
 * Text converter interface.
 */
struct TextConverter
{
    /**
     * Converts "in" text from specified "charset" to "out" UTF-8.
     */
    virtual int convert(const std::string& in, std::string& out, const std::string& charset)
        const = 0;
    virtual ~TextConverter()
    {
    }
    static bool succeeded(int res)
    {
        return res == ICONV_OK;
    }
    static bool failed(int res)
    {
        return !succeeded(res);
    }
};

/**
 * Normal text convertor to UTF-8 via iconv.
 */
struct NormalTextConverter : public TextConverter
{
    virtual int convert(const std::string& in, std::string& out, const std::string& charset) const
    {
        return Iconv::recode(charset, "utf-8", in, out, true);
    }
};

/**
 * Special implementation for the UTF-8 to UTF-8 conversion case.
 */
struct UtfTextConverter : public TextConverter
{
    virtual int convert(const std::string& in, std::string& out, const std::string& /* charset */)
        const
    {
        out = pystrutils::get_validated_utf8(in);
        return ICONV_OK;
    }
};

/**
 * Parent for the double coded text converters' implementations
 */
struct DoubleTextConverter : public TextConverter
{
    int recodeDouble(
        const std::string& in,
        std::string& out,
        const std::string& codepage,
        const std::string& middleCodepage) const
    {
        int res;
        if ((res = Iconv::recode(codepage, middleCodepage, in, out, true)) != ICONV_OK)
        {
            return res;
        }
        std::string buffer;
        std::swap(buffer, out);
        if ((res = Iconv::recode(codepage, "utf-8", buffer, out, true)) != ICONV_OK)
        {
            return res;
        }
        return ICONV_OK;
    }
};

/**
 * Implementation for the double coded koi-8 text
 */
struct KoiKoiTextConverter : public DoubleTextConverter
{
    virtual int convert(const std::string& in, std::string& out, const std::string&) const
    {
        return recodeDouble(in, out, "koi8-r", "cp1251");
    }
};

/**
 * Implementation for the double coded cp1251 text
 */
struct WinWinTextConverter : public DoubleTextConverter
{
    virtual int convert(const std::string& in, std::string& out, const std::string&) const
    {
        return recodeDouble(in, out, "cp1251", "koi8-r");
    }
};

}

#endif /* CHARACTER_CONVERTER_H_ */
