#include "utfizer.h"

#include <boost/algorithm/string.hpp>

namespace yimap {

const CharsetTraitsMap UTFizer::map;

Recognizer::TextTraits UTFizer::recognize(const std::string& text) const
{
    return profiler ? recognizeWithProfiling(text) : Recognizer::recognize(text);
}

Recognizer::TextTraits UTFizer::recognizeWithProfiling(const std::string& text) const
{
    pa::pause_stimer_t timer;
    Recognizer::TextTraits retval = Recognizer::recognize(text);
    profiler->add(pa::etc, "", "recognize", "", timer.stop());
    return retval;
}

Recognizer::CodesEnum UTFizer::recognizeEncoding(const std::string& text) const
{
    return profiler ? recognizeEncodingWithProfiling(text) : Recognizer::recognizeEncoding(text);
}

Recognizer::CodesEnum UTFizer::recognizeEncodingWithProfiling(const std::string& text) const
{
    pa::pause_stimer_t timer;
    Recognizer::CodesEnum retval = Recognizer::recognizeEncoding(text);
    profiler->add(pa::etc, "", "recognizeEncoding", "", timer.stop());
    return retval;
}

int UTFizer::convert(const CharsetTraitsMap::CharsetTraits& traits, std::string& text) const
{
    std::string out;
    const int res = traits.convertor.convert(text, out, traits.charset);
    if (succeeded(res))
    {
        std::swap(text, out);
    }
    return res;
}

int UTFizer::convert(const std::string& charset, std::string& text) const
{
    int result = convert(map.getCharsetTraits(charset), text);
    if (failed(result))
    {
        result = convert(recognizeEncoding(text), text);
    }
    return result;
}

int UTFizer::convert(Recognizer::CodesEnum code, std::string& text) const
{
    return convert(map.getCharsetTraits(code), text);
}

bool UTFizer::untrustedCharset(const std::string& charset) const
{
    using boost::algorithm::iequals;
    return iequals(charset, "us-ascii") || iequals(charset, "iso-8859-1") || charset.empty();
}

int UTFizer::utfize(const std::string& charset, std::string& text) const
{
    if (untrustedCharset(charset))
    {
        return convert(recognizeEncoding(text), text);
    }
    return convert(charset, text);
}

int UTFizer::utfize(const std::string& charset, std::string& text, Recognizer::LangsEnum& lang)
    const
{
    Recognizer::TextTraits traits(recognize(text));
    lang = traits.lang;
    if (untrustedCharset(charset))
    {
        return convert(traits.code, text);
    }
    return convert(charset, text);
}

Recognizer::LangsEnum utfizeStringDetectLang(std::string& content, const std::string& charset)
{
    Recognizer::LangsEnum lang = Recognizer::LANG_UNK;
    if (charset.empty())
    {
        UTFizer::process("utf-8", content, lang);
    }
    else
    {
        int result = UTFizer::process(charset, content, lang);
        if (UTFizer::failed(result))
        {
            throw UTFizerException(
                "could not utfize string. charset = " + charset + "; content = " + content);
        }
    }
    return lang;
}

void utfizeString(std::string& content, const std::string& charset)
{
    if (charset.empty())
    {
        UTFizer::process("utf-8", content);
    }
    else
    {
        int result = UTFizer::process(charset, content);
        if (UTFizer::failed(result))
        {
            throw UTFizerException(
                "could not utfize string. charset = " + charset + "; content = " + content);
        }
    }
}

} /* namespace yimap */
