#pragma once

#include <backend/mbody/util/character_traits.h>
#include <backend/mbody/util/recognizer_wrapper.h>
#include <pa/async.h>

namespace yimap {

class UTFizerException : public std::runtime_error
{
public:
    UTFizerException(const std::string& msg) : std::runtime_error(msg)
    {
    }
};

class UTFizer
{
public:
    typedef pa::async_profiler Profiler;
    static bool succeeded(int res)
    {
        return TextConverter::succeeded(res);
    }
    static bool failed(int res)
    {
        return TextConverter::failed(res);
    }

    static int process(const std::string& charset, std::string& text)
    {
        Profiler profiler;
        return UTFizer(profiler).utfize(charset, text);
    }
    static int process(const std::string& charset, std::string& text, Recognizer::LangsEnum& lang)
    {
        Profiler profiler;
        return UTFizer(profiler).utfize(charset, text, lang);
    }
    int utfize(const std::string& charset, std::string& text) const;
    int utfize(const std::string& charset, std::string& text, Recognizer::LangsEnum& lang) const;
    Recognizer::CodesEnum recognizeEncoding(const std::string& text) const;

    UTFizer() : profiler(0)
    {
    }
    UTFizer(Profiler& profiler) : profiler(&profiler)
    {
    }

private:
    bool untrustedCharset(const std::string& charset) const;
    int convert(Recognizer::CodesEnum code, std::string& text) const;
    int convert(const std::string& charset, std::string& text) const;
    int convert(const CharsetTraitsMap::CharsetTraits& traits, std::string& text) const;
    Recognizer::TextTraits recognize(const std::string& text) const;
    Recognizer::TextTraits recognizeWithProfiling(const std::string& text) const;
    Recognizer::CodesEnum recognizeEncodingWithProfiling(const std::string& text) const;
    static const CharsetTraitsMap map;

    Profiler* profiler;
};

Recognizer::LangsEnum utfizeStringDetectLang(std::string& content, const std::string& charset);
void utfizeString(std::string& content, const std::string& charset);

} // namespace yimap
