#ifndef _YIMAP_BACKEND_SHARPEI_RESOLVER_H
#define _YIMAP_BACKEND_SHARPEI_RESOLVER_H

#include <common/types.h>

#include <yplatform/util/sstream.h>
#include <sharpei_client/sharpei_client.h>
#include <ymod_httpclient/cluster_client.h>

namespace yimap { namespace backend {

using macs::pg::SharpeiParams;
using namespace sharpei::client::http;

class SharpeiHttpClient : public sharpei::client::http::HttpClient
{
public:
    SharpeiHttpClient() : client(yplatform::find<yhttp::cluster_client>("sharpei_client"))
    {
    }

    void aget(
        const Address& /*addr*/,
        Timeout /*timeout*/,
        const std::string& method,
        const Arguments& args,
        const Headers& headers,
        ResponseHandler handler,
        bool /*keepAlive*/,
        const std::string& /* requestId */) const override
    {
        auto url = makeUrl(method, args);
        auto hdrs = makeHeaders(headers);
        auto request = yhttp::request::GET(url, hdrs);
        call(std::move(request), std::move(handler));
    }

    void apost(
        const Address& /*addr*/,
        Timeout /*timeout*/,
        const std::string& method,
        const Arguments& args,
        const Headers& headers,
        const std::string& data,
        ResponseHandler handler,
        bool /*keepAlive*/,
        const std::string& /* requestId */) const override
    {
        auto url = makeUrl(method, args);
        auto hdrs = makeHeaders(headers);
        auto request = yhttp::request::POST(url, hdrs, std::string(data));
        call(std::move(request), std::move(handler));
    }

protected:
    void call(yhttp::request request, ResponseHandler handler) const
    {
        client->async_run(
            boost::make_shared<Context>(),
            std::move(request),
            [handler](const ErrorCode& err, const yhttp::response& res) {
                handler(
                    err,
                    err ? Response() :
                          Response{ static_cast<unsigned>(res.status), std::move(res.body) });
            });
    }

    std::string makeUrl(const std::string& method, const Arguments& args) const
    {
        std::vector<string> pairs;
        for (auto arg : args)
        {
            pairs.push_back(arg.first + "=" + boost::join(arg.second, ","));
        }
        stringstream urlStream;
        urlStream << method;
        urlStream << "?" << boost::join(pairs, "&");
        return urlStream.str();
    }

    std::string makeHeaders(const Headers& headers) const
    {
        std::string result;
        yplatform::sstream str(result);
        for (const auto& header : headers)
        {
            str << header.first << ": " << boost::join(header.second, ", ") << "\r\n";
        }
        return result;
    }

    boost::shared_ptr<yhttp::cluster_client> client;
};

SharpeiParams readSharpeiParams(const yplatform::ptree& xml)
{
    sharpei::client::http::Address sharpeiAddress = { xml.get("sharpei.host", ""),
                                                      xml.get<unsigned>("sharpei.port", 80) };
    if (sharpeiAddress.host.empty()) throw std::runtime_error("Invalid sharpei params");

    sharpei::client::Settings sharpeiSettings;
    sharpeiSettings.sharpeiAddress = sharpeiAddress;
    sharpeiSettings.timeout =
        sharpei::client::http::Timeout(xml.get<unsigned>("sharpei.timeout", 300));
    sharpeiSettings.retries = xml.get<unsigned>("sharpei.retries", 3u);
    sharpeiSettings.keepAlive = xml.get<int>("sharpei.keepalive", 0);

    return SharpeiParams{ std::make_shared<SharpeiHttpClient>(),
                          sharpeiSettings,
                          { xml.get("sharpei.pg_user", ""), "" } };
}

} // namespace backend
} // namespace yimap

#endif // _YIMAP_BACKEND_SHARPEI_RESOLVER_H
