#include "session.h"

namespace yimap { namespace server {

Session::Session(
    yplatform::net::base_service* service,
    const ServerSettings& settings,
    std::unique_ptr<ProtocolSessionBase>&& sessionImpl)
    : base_t(service, settings)
    , sessionImpl(std::move(sessionImpl))
    , sessionLogger(this->sessionImpl->getLogger())
    , baseService(service)
    , readq_(new buffer_t)
    , read_timer_(*service->get_io())
    , write_timer_(*service->get_io())
{
}

Session::~Session()
{
    if (settings().trafficLogEnabled)
    {
        flushTrafficLog();
    }
}

void Session::core_open()
{
    auto self = shared_from_this();
    postOnStrand([self, this]() {
        isSessionOpen = true;
        sessionLogger.logDebugNet() << "=== core_open()";
        if (this->settings().force_ssl)
            start_tls(boost::bind(&Session::handle_tls_handshake, shared_from_this(), _1));
        else
            start_session();
    });
}

void Session::start_session()
{
    auto self = shared_from_this();
    postOnStrand([self, this]() {
        SessionInfo sessionInfo(raw_socket().local_endpoint(), raw_socket().remote_endpoint());
        sessionImpl->init(sessionInfo);
        sessionLogger.logDebugNet() << " === start_session()";
        sessionImpl->greetings();
        startReader(false, 1);
    });
}

void Session::startTls()
{
    needStartTls = true;
    if (isReading)
    {
        cancelOperations();
    }
    else
    {
        isReading = true;
        start_tls(boost::bind(&Session::handle_tls_handshake, shared_from_this(), _1));
    }
}

void Session::handle_tls_handshake(const boost::system::error_code& e)
{
    auto self = shared_from_this();
    postOnStrand([e, this, self]() {
        needStartTls = false;
        isReading = false;

        if (e)
        {
            sessionLogger.logEvent() << " handle_tls_handshake: error: " << e.message();
            cancel_operations();
            return;
        }

        sessionImpl->onTls(true, getSSLChiphers());

        if (this->settings().force_ssl)
        {
            start_session();
        }
        else
        {
            sessionLogger.logEvent() << " handle_tls_handshake: starting reader";
            isSessionOpen = true;
            asyncStartReader(true);
        }
    });
}

void Session::cancelOperations()
{
    sessionLogger.logDebugNet() << " cancelOperations";
    cancel_operations();
}

void Session::handle_tls_shutdown()
{
    sessionImpl->onTls(false, "");
    sessionLogger.logEvent() << " handle_tls_shutdown: starting reader";
    asyncStartReader(true);
}

string Session::getSSLChiphers()
{
    auto sslStream = stream().get_ssl_stream();
    if (!sslStream)
    {
        return "error: no ssl stream";
    }

    SSL* ssl = sslStream->native_handle();
    const SSL_CIPHER* cipher = SSL_get_current_cipher(ssl);
    int ssl_algbits = 0;
    int ssl_usebits = SSL_CIPHER_get_bits(cipher, &ssl_algbits);

    ostringstream chipherStream;
    chipherStream << SSL_get_version(ssl) << " with cipher " << SSL_CIPHER_get_name(cipher) << " ("
                  << ssl_usebits << "/" << ssl_algbits << " bits)";
    return chipherStream.str();
}

//-----------------------------------------------------------------------------
// Writing data

yplatform::net::streamer_wrapper Session::client_stream()
{
    auto sess = shared_from_this();
    yplatform::net::streamer_wrapper wrp(new yplatform::net::streamer<Session>(sess));
    return wrp;
}

ClientStream Session::clientStream()
{
    auto s = shared_from_this();
    return ClientStream(s);
}

void Session::send_client_stream(yplatform::net::buffers::const_chunk_buffer const& s)
{
    sendClientStream(s, false, "");
}

void Session::sendClientStream(
    const yplatform::net::buffers::const_chunk_buffer& s,
    bool skip_log,
    const string& debugRecord)
{
    auto self = shared_from_this();
    postOnStrand([s, skip_log, debugRecord, this, self]() {
        if (!raw_socket().is_open()) return;

        writeq_.push(s);

        sessionImpl->onSendClientStream(s, skip_log, debugRecord);
        startWriter();
    });
}

void Session::startWriter()
{
    if (!isWriting)
    {
        isWriting = true;
        try
        {
            bool flush_ok = !writeq_.flush(settings().limits.max_output_chunk);
            if (!flush_ok)
            {
                isWriting = false;
                return;
            }

            doWrite();
        }
        catch (...)
        {
            sessionLogger.logEvent() << " start_writer: ERROR during sending data to client";
            isWriting = false;
        }
    }
}

void Session::doWrite()
{
    resetTimer(write_timer_, settings().write_timeout);
    read_timer_.cancel();

    async_write(
        writeq_.send_queue(),
        std::bind(
            &Session::handleWrite,
            shared_from_this(),
            std::placeholders::_1,
            std::placeholders::_2));
}

void Session::handleWrite(const boost::system::error_code& e, size_t bytes)
{
    auto self = shared_from_this();
    postOnStrand([e, bytes, this, self]() {
        isWriting = false;

        if (e)
        {
            handleWriteError(e);
            return;
        }

        writeq_.consume(bytes);

        if (settings().trafficLogEnabled)
        {
            writeTraffic += bytes;
            if (writeTraffic >= 2048)
            {
                flushTrafficLog();
            }
        }

        startWriter();
    });
}

void Session::handleWriteError(const boost::system::error_code& e)
{
    sessionLogger.logEvent() << " handle_write: error: " << e.message();
    sessionImpl->onWriteError(e);
}

//-----------------------------------------------------------------------------
// Reading data

void Session::asyncStartReader(bool forceStart, std::size_t atleast)
{
    strand().post(std::bind(&Session::startReader, shared_from_this(), forceStart, atleast));
}

void Session::startReader(bool forceStart, std::size_t atleast)
{

    if (shouldShutdown() && !forceStart) return;

    if (isReading) return;

    isReading = true;
    try
    {
        doRead();
    }
    catch (...)
    {
        // sessionLogger.logEvent() << " start_reader: ERROR during reading data from client";
    }
}

void Session::doRead()
{
    if (needStartTls) return;

    resetTimer(read_timer_, settings().read_timeout);
    write_timer_.cancel();

    std::size_t atleast = 1u;
    async_read(
        std::bind(
            &Session::handleRead, shared_from_this(), std::placeholders::_1, std::placeholders::_2),
        readq_->prepare(atleast),
        atleast);
}

void Session::handleRead(const boost::system::error_code& e, std::size_t bytes)
{
    auto self = shared_from_this();
    postOnStrand([e, bytes, this, self]() {
        read_timer_.cancel();

        if (needStartTls)
        {
            start_tls(boost::bind(&Session::handle_tls_handshake, shared_from_this(), _1));
            return;
        }

        isReading = false;

        if (e)
        {
            isSessionOpen = false;
            sessionImpl->onReadError(e);
            cancelOperations();
            return;
        }

        readq_->commit(bytes);
        sessionImpl->processData(string(readq_->begin(), readq_->end()));
        readq_->consume(bytes);
    });
}

void Session::resumeReading()
{
    sessionImpl->processData(string());
}

//-----------------------------------------------------------------------------
// Closing session

void Session::resetTimer(Timer& timer, yplatform::time_traits::duration timeout)
{
    timer.cancel();
    timer.expires_from_now(timeout);

    SessionWeakPtr weakSess = shared_from_this();
    timer.async_wait(yplatform::net::create_timer_handler([weakSess]() {
        auto that = weakSess.lock();
        if (that) that->asyncAutologout();
    }));
}

void Session::asyncAutologout()
{
    strand().post(std::bind(&Session::onAutologout, shared_from_this()));
}

void Session::onAutologout()
{
    sessionLogger.logEvent() << " onAutologoutTimer: closing session";
    sessionImpl->autologout();
    do_shutdown(true);
}

void Session::do_shutdown(bool graceful)
{
    if (shouldShutdown()) return;

    sessionImpl->onShutdown();

    isSessionOpen = false;
    setShutdown(true);

    read_timer_.cancel();
    write_timer_.cancel();

    try
    {
        if (graceful) raw_socket().shutdown(raw_socket_t::shutdown_receive);
        else
            raw_socket().close();
    }
    catch (std::exception const& err)
    {
        sessionLogger.logEvent() << " do_shutdown: std::exception: " << err.what();
    }
    catch (...)
    {
        sessionLogger.logEvent() << " do_shutdown: unknown exception";
    }
}

void Session::flushTrafficLog()
{
    if (writeTraffic > 0)
    {
        sessionLogger.logTraffic("write " + std::to_string(writeTraffic) + " bytes");
        writeTraffic = 0;
    }
}

//-----------------------------------------------------------------------------
// Timer managment

void Session::postOnStrand(VoidFunction hook)
{
    strand().post(hook);
}

void Session::changeTimeouts(ChangeTimeouts::Class newState)
{
    if (!Session::is_open()) return;

    local_settings().changeTimeouts(newState);
    resetTimer(read_timer_, settings().read_timeout);
    resetTimer(write_timer_, settings().write_timeout);
}

TimerPtr Session::createTimer(uint32_t milliseconds, VoidFunction hook)
{
    yplatform::time_traits::duration timeout = yplatform::time_traits::milliseconds(milliseconds);
    TimerPtr timer(new Timer(*baseService->get_io()));
    timer->expires_from_now(timeout);
    timer->async_wait(strand().wrap(yplatform::net::create_timer_handler(hook)));
    return timer;
}

//-----------------------------------------------------------------------------

ContextPtr Session::getContext()
{
    return sessionImpl->getContext();
}

Logger& Session::getLogger()
{
    return sessionImpl->getLogger();
}

//-----------------------------------------------------------------------------
// Network events

void Session::postEvent(NetworkEvent event)
{
    strand().post(std::bind(&Session::processEvent, shared_from_this(), event));
}

void Session::processEvent(NetworkEvent event)
{
    if (event->name == "ChangeTimeouts")
    {
        auto changeTimeoutsEvent = std::dynamic_pointer_cast<ChangeTimeouts>(event);
        if (changeTimeoutsEvent)
        {
            changeTimeouts(changeTimeoutsEvent->data);
        }
    }
    else if (event->name == "ShutdownSession")
    {
        do_shutdown(true);
    }
    else if (event->name == "StartReader")
    {
        asyncStartReader();
    }
    else if (event->name == "ResumeReading")
    {
        resumeReading();
    }
    else if (event->name == "StartTls")
    {
        startTls();
    }
}

//-----------------------------------------------------------------------------
// ClientSream

ClientSreamImpl::~ClientSreamImpl()
{
    try
    {
        auto data = std::move(outstream.str());
        if (data.empty()) return;
        // TODO: std move
        session->send_const_string(data);
    }
    catch (std::exception& e)
    {
        session->getLogger().logError() << "std::exception in ~ClientSreamImpl(): " << e.what();
    }
    catch (...)
    {
        session->getLogger().logError() << "Unknown exception in ~ClientSreamImpl()";
    }
}

} // namespace server
} // namespace yimap
