#ifndef _YIMAP_NETWORK_SESSION_H_
#define _YIMAP_NETWORK_SESSION_H_

#include <network/protocol_session_base.h>
#include <network/settings.h>

#include <yplatform/net/buffer_sequence.h>
#include <yplatform/net/server_session_strand.h>
#include <yplatform/zerocopy/streambuf.h>

#include <boost/system/error_code.hpp>

namespace yimap { namespace server {

typedef boost::mutex mutex_t;
typedef boost::unique_lock<mutex_t> lock_t;

class Session
    : public NetworkSession
    , public yplatform::net::server_session_strand<ServerSettings>
    , public boost::enable_shared_from_this<Session>
{
protected:
    typedef yplatform::net::server_session_strand<ServerSettings> base_t;
    typedef yplatform::net::output_buffer_sequence<yplatform::net::buffers::const_chunk_buffer>
        output_queue_t;

    typedef yplatform::zerocopy::streambuf buffer_t;
    typedef boost::shared_ptr<buffer_t> buffer_ptr;

public:
    Session(
        yplatform::net::base_service* service,
        const ServerSettings& settings,
        std::unique_ptr<ProtocolSessionBase>&& sessionImpl);
    virtual ~Session();

    bool is_open() const
    {
        return base_t::is_open() && isSessionOpen;
    }
    virtual bool isOpen() const
    {
        return is_open();
    }

    void core_open();
    void start_session();

    // streamer<session> client_stream();
    virtual yplatform::net::streamer_wrapper client_stream();
    virtual ClientStream clientStream();

    virtual void postEvent(NetworkEvent event);
    virtual void postOnStrand(VoidFunction);

    virtual TimerPtr createTimer(uint32_t milliseconds, VoidFunction hook);

    void changeTimeouts(ChangeTimeouts::Class newState);

    virtual void send_client_stream(const yplatform::net::buffers::const_chunk_buffer& s);
    virtual void send_client_stream2(const yplatform::net::buffers::const_chunk_buffer&, bool)
    {
    }
    virtual void sendClientStream(
        const yplatform::net::buffers::const_chunk_buffer& s,
        bool skip_log,
        const string& extraLogMessage);

    NetworkSessionPtr shared_base_from_this()
    {
        return this->shared_from_this();
    }

    virtual ContextPtr getContext();
    virtual Logger& getLogger();
    virtual std::string hostname()
    {
        return settings().host_name;
    };

protected:
    void processEvent(NetworkEvent event);

    void resetTimer(Timer& timer, yplatform::time_traits::duration timeout);
    void asyncAutologout();
    void onAutologout();

    void startWriter();
    void doWrite();
    void handleWrite(const boost::system::error_code& e, size_t bytes);
    void handleWriteError(boost::system::error_code const& e);

    void asyncStartReader(bool forceStart = false, std::size_t atleast = 1);
    void startReader(bool forceStart, std::size_t atleast);
    void doRead();
    void handleRead(const boost::system::error_code& e, std::size_t bytes);
    void resumeReading();

    void flushTrafficLog();

    void do_shutdown(bool graceful = true);
    bool shouldShutdown() const
    {
        return shutdown_;
    }
    void setShutdown(bool shut)
    {
        shutdown_ = shut;
    }

protected:
    void startTls();

    void handle_tls_handshake(boost::system::error_code const& e);
    void handle_tls_shutdown();
    string getSSLChiphers();

    void cancelOperations();
    void onCancelOperations();

protected:
    std::unique_ptr<ProtocolSessionBase> sessionImpl;
    Logger& sessionLogger;
    yplatform::net::base_service* baseService;

    // writer
    output_queue_t writeq_;
    // reader
    buffer_ptr readq_;

    size_t writeTraffic = 0;

    bool shutdown_ = false;
    bool needStartTls = false;

    bool isReading = false;
    bool isWriting = false;
    bool isSessionOpen = false;

    Timer read_timer_;
    Timer write_timer_;
};

template <class ProtocolSessionImpl>
class ProtocolSession : public Session
{
public:
    ProtocolSession(yplatform::net::base_service* service, const ServerSettings& settings)
        : Session(service, settings, std::move(makeImpl(settings)))
    {
    }

    std::unique_ptr<ProtocolSessionBase> makeImpl(const ServerSettings& settings)
    {
        SessionPtrFunction sessionHook = std::bind(&Session::shared_base_from_this, this);
        return std::unique_ptr<ProtocolSessionBase>(new ProtocolSessionImpl(settings, sessionHook));
    }
};

typedef boost::shared_ptr<Session> SessionPtr;
typedef boost::weak_ptr<Session> SessionWeakPtr;

}}

#endif // _YIMAP_NETWORK_SESSION_H_
