#pragma once

#include "session_fwd.h"

#include <common/config.h>
#include <common/context.h>

#include <yplatform/net/settings.h>

namespace yimap { namespace server {

struct ServerLimits
{
    std::size_t max_all_literal_size = 30 * yplatform::net::server_settings::MB;
    std::size_t max_command_size = 35 * yplatform::net::server_settings::MB;
    std::size_t max_literal_size = 30 * yplatform::net::server_settings::MB;
    std::size_t max_output_chunk = 10 * yplatform::net::server_settings::KB;
    std::size_t max_packet_size = 512;
    std::size_t max_pure_size = 8 * yplatform::net::server_settings::KB;
};

struct ServerSettings : public yplatform::net::server_settings
{
    bool force_ssl = false;
    AuthConfig acfg;

    ServerLimits limits;

    yplatform::net::time_duration autologout = yplatform::time_traits::duration::max();
    yplatform::net::time_duration autologout_init = yplatform::time_traits::duration::max();
    yplatform::net::time_duration autologout_idle = yplatform::time_traits::duration::max();

    std::string host_name = HostId::create().hostName;

    bool trafficLogEnabled = false;

    virtual void parse_ptree(const boost::property_tree::ptree& data)
    {
        boost::optional<const boost::property_tree::ptree&> acfg_data =
            data.get_child_optional("auth_cfg");

        force_ssl = data.get("force_ssl", 0);
        if (acfg_data) acfg.parse_ptree(acfg_data.get());

        // Default autologout time is 40 sec.
        int autologout_milsec = data.get("autologout", 40000);
        int autologout_init_milsec = data.get("autologout_init", autologout_milsec);
        int autologout_idle_milsec = data.get("autologout_idle", autologout_milsec);

        if (autologout_milsec >= 0)
            autologout = yplatform::time_traits::milliseconds(autologout_milsec);
        if (autologout_init_milsec >= 0)
            autologout_init = yplatform::time_traits::milliseconds(autologout_init_milsec);
        if (autologout_idle_milsec >= 0)
            autologout_idle = yplatform::time_traits::milliseconds(autologout_idle_milsec);

        limits.max_all_literal_size = data.get("max_all_literal_size", limits.max_all_literal_size);
        limits.max_command_size = data.get("max_command_size", limits.max_command_size);
        limits.max_literal_size = data.get("max_literal_size", limits.max_literal_size);
        limits.max_output_chunk = data.get("max_output_chunk", limits.max_output_chunk);
        limits.max_packet_size = data.get("max_packet_size", limits.max_packet_size);
        limits.max_pure_size = data.get("max_pure_size", limits.max_pure_size);

        trafficLogEnabled = data.get("traffic_log_enabled", false);

        yplatform::net::server_settings::parse_ptree(data);
        changeTimeouts(ChangeTimeouts::Init);
    }

    void changeTimeouts(ChangeTimeouts::Class newState)
    {
        switch (newState)
        {
        case ChangeTimeouts::Init:
            read_timeout = write_timeout = autologout_init;
            break;
        case ChangeTimeouts::Idle:
            read_timeout = autologout_idle;
            break;
        case ChangeTimeouts::Exit:
            read_timeout = write_timeout = yplatform::time_traits::milliseconds(0);
            break;
        case ChangeTimeouts::Respond:
            read_timeout = write_timeout = yplatform::time_traits::duration::max();
            break;
        default:
            read_timeout = write_timeout = autologout;
        }
    }
};

} // namespace server
} // namespace yimap
