#include "command.h"

#include <boost/algorithm/string/join.hpp>

namespace ypop {

using yimap::server::ShutdownSession;

PopCommand::PopCommand(
    const request_t& req,
    session_context_ptr context,
    NetworkSessionPtr ostr,
    processor_stats_ptr stats)
    : context(context), req(req), ostr(ostr)
{
    statistic = context->create_module_data<task_context>("pop_processor");
    statistic->set_module_stat(stats);
    command = req->empty() ? "" : req->at(0);
    context->command = command;
}

std::string PopCommand::to_log_string(bool log_cmd) const
{
    std::ostringstream os;
    os << "client=" << context->sessionInfo.remote_address << ':'
       << context->sessionInfo.remote_port;
    os << " state=";
    os << context->state_name();

    if (context->state != ypop::pop_context::START) os << " user=" << context->login;
    if (log_cmd && !req->empty()) os << " cmd=" << (*req)[0];
    return os.str();
}

void PopCommand::remember()
{
    auto str = (*req)[0];
    if (str == "pass") return;

    for (size_t i = 1; i < req->size(); i++)
    {
        str += " " + (*req)[i];
    }
    context->addHistoryEntry(std::move(str));
}

void PopCommand::rememberResponse(std::string&& resp)
{
    std::ostringstream respStream;
    respStream << command << "<" << resp << ">";
    context->addHistoryEntry(respStream.str());
}

void PopCommand::out_statistic()
{
    std::ostringstream statResponse;
    statResponse << context->messages->good_number << " " << context->messages->good_size;

    context->addHistoryEntry("stat<" + statResponse.str() + ">");
    ostr->clientStream() << "+OK " << statResponse.str() << "\r\n";
    prom.set(ProcessorResult::SUCCESS);
}

void PopCommand::out_server_error()
{
    ostr->clientStream() << "-ERR [SYS/TEMP] internal server error." << get_uniq_id_ending()
                         << "\r\n";
    prom.set(ProcessorResult::FAILURE);
}

void PopCommand::out_mailbox_busy()
{
    ostr->clientStream() << "-ERR [SYS/TEMP] Mailbox busy or something nasty happened..."
                         << get_uniq_id_ending() << "\r\n";
    context->state = session_context_t::QUIT;
    statistic->stats_login(task_context::api_error);
    prom.set(ProcessorResult::FAILURE);
}

void PopCommand::out_incorrect_command(const std::string& usage)
{
    context->addHistoryEntry(boost::join(*req, " "));

    if (usage.empty())
    {
        ostr->clientStream() << "-ERR What?" << get_uniq_id_ending() << "\r\n";

        context->addHistoryEntry((req->empty() ? "" : req->at(0)) + "<-ERR What?>");
    }
    else
    {
        ostr->clientStream() << "-ERR usage: " << usage << "." << get_uniq_id_ending() << "\r\n";

        context->addHistoryEntry(usage + "<-ERR usage>");
    }
    prom.set(ProcessorResult::FAILURE);
}

void PopCommand::out_incorrect_msgid()
{
    ostr->clientStream() << "-ERR message does not exist or deleted." << get_uniq_id_ending()
                         << "\r\n";
    prom.set(ProcessorResult::FAILURE);
}

void PopCommand::out_goobye()
{
    context->state = session_context_t::QUIT;
    ostr->clientStream() << "+OK shutting down.\r\n";
    prom.set(ProcessorResult::SUCCESS);
    ostr->postEvent(std::make_shared<ShutdownSession>());
}

} // namespace ypop
