#include <processor/pop_backend.h>
#include <processor/processor_impl.h>
#include <processor/message.h>

#include <backend/user_journal_types.h>

#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/url_encode.h>
#include <yplatform/find.h>

#include <boost/property_tree/json_parser.hpp>

namespace ypop {

namespace ph = std::placeholders;

void processor_impl::cmd_stls(pop_args_ptr args)
{
    if (args->context->tls_on)
    {
        args->ostr->clientStream()
            << "-ERR Command not permitted when TLS active" << args->get_uniq_id_ending() << "\r\n";
        args->prom.set(ProcessorResult::FAILURE);
    }
    else
    {
        args->ostr->clientStream() << "+OK Begin TLS negotiation\r\n";
        args->prom.set(ProcessorResult::START_TLS);
    }
}

void processor_impl::cmd_user(pop_args_ptr args)
{
    if (args->req->size() != 2) return args->out_incorrect_command("user <username>");

    args->context->login = (*args->req)[1];
    args->context->state = session_context_t::NAME;

    if (!checkSSL(args)) return;

    args->ostr->clientStream() << "+OK password, please.\r\n";
    args->prom.set(ProcessorResult::SUCCESS);
}

bool processor_impl::checkSSL(pop_args_ptr args)
{
    if (!settings_->ssl_only) return true;
    if (args->context->tls_on) return true;

    service_log_error(*args, "LOGIN error: untrusted connection for user " << args->context->login);
    args->ostr->clientStream()
        << "-ERR [AUTH] Working without SSL/TLS encryption is not allowed. "
        << "Please visit https://yandex.ru/support/mail-new/mail-clients/ssl.html "
        << args->get_uniq_id_ending() << "\r\n";
    args->prom.set(ProcessorResult::FAILURE);
    return false;
}

void processor_impl::cmd_pass(pop_args_ptr args)
{
    if (args->req->size() < 2) return args->out_incorrect_command("pass <password>");

    string password((*args->req)[1]);
    {
        // From http://www.ietf.org/rfc/rfc1939
        // "Since the PASS command has exactly one argument, a POP3
        //  server may treat spaces in the argument as part of the
        //  password, instead of as argument separators."
        // So we should merge all arguments to get long password
        for (size_t i = 2; i < args->req->size(); i++)
        {
            password += " ";
            password += (*args->req)[i];
        }

        boost::crc_32_type crc_generator;
        crc_generator.process_bytes(password.data(), password.size());
        args->context->passwordCrc = crc_generator();
    }

    ymod_blackbox::request areq(
        args->context,
        args->context->login,
        password,
        { args->context->sessionInfo.remote_address, args->context->sessionInfo.remote_port },
        "");
    areq.no_password = args->context->auth_cfg.allow_xlogin;
    areq.sid = args->context->auth_cfg.sid;
    areq.service = args->context->auth_cfg.service;

    yplatform::future::future<ymod_blackbox::response> aresp =
        yplatform::find<ymod_blackbox::auth>("auth")->authenticate(areq);

    aresp.add_callback(
        boost::bind(&processor_impl::blackbox_cb, get_shared_from_this(), args, aresp));
}

void processor_impl::blackbox_cb(pop_args_ptr args, BlackboxPromise authPromise)
{
    ymod_blackbox::response aresp;
    string errorPrivate;
    string errorPublic;
    bool authOk = checkAuthError(args, authPromise, aresp, errorPrivate, errorPublic);
    if (!authOk)
    {
        service_log_error(
            *args,
            "LOGIN: "
                << " result=FAIL module=blackbox"
                << " password_crc=" << args->context->passwordCrc << " error=" << aresp.error
                << " karma=" << aresp.karma << " karma_status=" << aresp.karma_status
                << " ban_time=" << aresp.ban_time << " reason='" << errorPrivate << " ("
                << aresp.err_verbose << ")'");

        args->context->state = session_context_t::QUIT;
        args->ostr->clientStream()
            << "-ERR " << errorPublic << args->get_uniq_id_ending() << "\r\n";

        args->prom.set(ProcessorResult::FAILURE_CLOSE);
        return;
    }

    {
        LOCK_POP_CONTEXT(*args->context);
        args->context->suid = aresp.suid;
        args->context->uid = aresp.uid;
        args->context->storage = aresp.storage;
        args->context->email = aresp.email;
    }

    loadSettingsAsync(args);
}

bool processor_impl::checkAuthError(
    pop_args_ptr args,
    BlackboxPromise& authPromise,
    ymod_blackbox::response& result,
    string& errorPrivate,
    string& errorPublic)
{
    errorPrivate = "";
    if (authPromise.has_exception())
    {
        errorPrivate = get_exception_reason(authPromise);
        errorPublic = "[SYS/TEMP] Service unavailable.";
        args->statistic->stats_login(task_context::bb_error);
        return false;
    }
    else
    {
        result = authPromise.get();
        errorPrivate = result.err_str;
        errorPublic = "[AUTH] login failure or POP3 disabled, try later.";
        if (settings_->enable_bb_block && result.user_blocked)
        {
            errorPrivate = "user blocked in blackbox";
        }
        else if (result.suid.empty() || result.storage.empty())
        {
            errorPrivate = "blackbox return empty user data";
        }
        else if (result.storage != "pg")
        {
            errorPrivate = "user not in 'pg': " + result.storage;
        }
        args->statistic->stats_login(task_context::rejected);
    }
    return errorPrivate.empty();
}

void processor_impl::loadSettingsAsync(pop_args_ptr args)
{
    args->ostr->postOnStrand(
        std::bind(&processor_impl::loadSettings, get_shared_from_this(), args));
}

void processor_impl::loadSettings(pop_args_ptr args)
{
    static const std::string settingsList = "enable_pop\r"
                                            "pop3_makes_read\r"
                                            "pop_spam_enable\r"
                                            "pop_spam_subject_mark_enable";
    auto ctx = args->context;
    auto cb =
        std::bind(&processor_impl::settingsCallback, get_shared_from_this(), args, ph::_1, ph::_2);
    auto request = yhttp::request::GET(
        "/get" +
        yhttp::url_encode({ { "uid", ctx->uid },
                            { "suid", ctx->suid },
                            { "mdb", ctx->storage },
                            { "settings_list", settingsList },
                            { "db_role", "replica" },
                            { "format", "json" } }));
    auto client = yplatform::find<yhttp::cluster_client>("settings_client");
    client->async_run(
        ctx, std::move(request), [ctx, cb](boost::system::error_code ec, yhttp::response response) {
            using error_code = ymod_httpclient::http_error::code;
            try
            {
                if (ec)
                {
                    TASK_LOG(ctx, error) << "get settings error: " << ec.message();
                    return cb(ec, UserSettings());
                }

                if (response.status != 200)
                {
                    TASK_LOG(ctx, error) << "get settings error: status= " << response.status
                                         << " reason=" << response.reason;
                    return cb(error_code::server_status_error, UserSettings());
                }

                std::stringbuf buf(response.body);
                std::istream stream(&buf);
                yplatform::ptree ptree;
                boost::property_tree::read_json(stream, ptree);
                UserSettings settings;
                auto profileSettings = ptree.get_child("settings.profile.single_settings");
                settings.enablePop = profileSettings.get("enable_pop", settings.enablePop);
                settings.markRead = profileSettings.get("pop3_makes_read", settings.markRead);
                settings.spamEnable = profileSettings.get("pop_spam_enable", settings.spamEnable);
                settings.spamSubjectMarkEnable = profileSettings.get(
                    "pop_spam_subject_mark_enable", settings.spamSubjectMarkEnable);
                cb(ec, settings);
            }
            catch (const std::exception& e)
            {
                TASK_LOG(ctx, error) << "get settings exception: " << e.what();
                cb(error_code::parse_response_error, UserSettings());
            }
        });
}

void processor_impl::settingsCallback(
    pop_args_ptr args,
    const boost::system::error_code& ec,
    const UserSettings& settings)
{
    if (ec)
    {
        service_log_error(
            *args,
            "Failed to load user settings: " << ec.message() << ". Fallback to default values.");
    }
    args->ostr->postOnStrand(
        std::bind(&processor_impl::processSettings, get_shared_from_this(), args, settings));
}

void processor_impl::processSettings(pop_args_ptr args, const UserSettings& settings)
{
    {
        LOCK_POP_CONTEXT(*args->context);
        args->context->settings = settings;
    }

    if (!args->context->settings.enablePop)
    {
        TASK_LOG(args->context, info) << " Login failed: pop3 is disabled in settings";
        args->ostr->clientStream() << "-ERR [AUTH] login failure or POP3 disabled, try later."
                                   << args->get_uniq_id_ending() << "\r\n";
        args->context->state = session_context_t::QUIT;
        args->statistic->stats_login(task_context::disabled);
        args->prom.set(ProcessorResult::FAILURE_CLOSE);
        return;
    }

    args->context->pop3Backend = std::make_shared<PgMessageLoader>(*args->context, settings_);
    loadMessages(args);
}

void processor_impl::loadMessages(pop_args_ptr args)
{
    try
    {
        auto futureMessages = args->context->pop3Backend->loadUidl();
        auto handler = boost::bind(
            &processor_impl::handleMessages, get_shared_from_this(), args, futureMessages);
        futureMessages.add_callback([this, args, handler] { args->ostr->postOnStrand(handler); });
    }
    catch (const std::exception& e)
    {
        service_log_error(
            *args,
            "MACS error: failed to load messages from " << args->context->storage << ": "
                                                        << e.what());
        return args->out_mailbox_busy();
    }
    catch (...)
    {
        service_log_error(*args, "MACS error: failed to load messages " << args->context->storage);
        return args->out_mailbox_busy();
    }
}

void processor_impl::handleMessages(pop_args_ptr args, FutureMessageList messages)
{
    try
    {
        args->context->messages = messages.get();
    }
    catch (const std::exception& e)
    {
        service_log_error(
            *args,
            "MACS error: failed to load messages from " << args->context->storage << ": "
                                                        << e.what());
        return args->out_mailbox_busy();
    }
    catch (...)
    {
        service_log_error(*args, "MACS error: failed to load messages " << args->context->storage);
        return args->out_mailbox_busy();
    }
    completeLogin(args);
}

void processor_impl::completeLogin(pop_args_ptr args)
{
    if (settings_->loginSettings.userJournalLogEnable)
    {
        journalOperation<user_journal_types::Authorization>(
            args->context,
            user_journal::parameters::id::state(std::string()),
            user_journal::parameters::id::affected(0u));
    }

    TASK_LOG(args->context, info) << "User login: " << args->context->login
                                  << ", storage: " << args->context->storage
                                  << ", suid: " << args->context->suid
                                  << ", ip: " << args->context->sessionInfo.remote_address
                                  << ", email: " << args->context->email
                                  << ", ssl: " << args->context->sslStatus << " connected.";

    // TYPED_TABLE_LOG(::yplatform::log::detail::get_logger_from_ctx(args->context), "pop3_auth")
    // << log::time_attr
    //    << log::make_attr("session", args->context->uniq_id())
    //    << log::make_attr("ip",      args->context->sessionInfo.remote_address)
    //    << log::make_attr("login",   args->context->login)
    //    << log::make_attr("storage", args->context->storage)
    //    << log::make_attr("uid",     args->context->uid)
    //    << log::make_attr("suid",    args->context->suid)
    //    << log::make_attr("email",   args->context->email)
    //    << log::make_attr("ssl",     args->context->sslStatus);

    args->context->state = session_context_t::TRANS;
    args->statistic->stats_login(task_context::granted);
    args->out_statistic();
}

} // namespace ypop
