#include <processor/pop_backend.h>
#include <common/config.h>
#include <backend/mbody/message_loader.h>
#include <backend/user_journal_types.h>

#include <processor/processor_impl.h>
#include <processor/message.h>

#include <yplatform/util.h>

using yimap::StringPtr;

namespace ypop {

const string POP_HEADER_FOLDER_NAME = "X-Yandex-FolderName: ";
const string POP_HEADER_SPAM_FLAG = "X-Oborona-Spam-Flag: YES";
const string POP_HEADER_SPAM_SUBJECT = "[OBORONA-SPAM] ";

class PopMessageHandler : public yimap::mbody::MessageDataHandler
{
public:
    PopMessageHandler(pop_context_ptr ctx, const message_entry& msg, int lines)
        : popContext(ctx)
        , lines_(lines)
        , current_lines_(-1)
        , receive_stopped_(false)
        , subject_processed_(false)
        , message(msg)
        , current_list_(&headers_)
        , hasLog_(false)
    {
    }

    virtual ~PopMessageHandler()
    {
    }

    void onMessage(StringPtr messageData, size_t) override
    {
        parseMessage(messageData->c_str(), messageData->size());
    }

    void parseMessage(const char* data, unsigned long long size)
    {
        if (receive_stopped_ || size == 0) return;
        if (current_lines_ >= 0 && lines_ >= 0 && ++current_lines_ >= lines_)
        {
            receive_stopped_ = true;
            return;
        }
        if (current_list_->empty()) current_list_->push_back(StringPtr(new string));
        const char* i = data;
        const char* end = data + size;
        for (; i < end; ++i)
        {
            if (*i != '\n') continue;

            string& back_str = *current_list_->back();
            if (back_str.empty() && *data == '.') back_str.push_back('.');
            back_str.reserve(current_list_->back()->size() + i - data + 10);
            back_str.append(data, ++i);

            // Correct "\n" to "\r\n". We know that back_str has at least 1 characte - '\n'
            if (back_str.size() == 1 || back_str[back_str.size() - 2] != '\r')
            {
                back_str[back_str.size() - 1] = '\r';
                back_str.push_back('\n');
            }

            data = i;

            if (current_lines_ >= 0 && lines_ >= 0 && ++current_lines_ >= lines_)
            {
                receive_stopped_ = true;
                return;
            }

            if (current_lines_ < 0 && message.is_spam && popContext->settings.spamSubjectMarkEnable)
            {
                // find Subject: XXX and replace to Subject: [OBORONA-SPAM] XXX
                if (yplatform::util::iequals(
                        boost::make_iterator_range(back_str.begin(), back_str.begin() + 7),
                        "subject"))
                {
                    string::iterator i = back_str.begin() + 7;
                    while (i != back_str.end() && ::isspace(*i))
                        ++i;
                    if (i != back_str.end() && *i == ':')
                    {
                        do
                        {
                            ++i;
                        } while (i != back_str.end() && ::isspace(*i));
                        back_str.insert(
                            i, POP_HEADER_SPAM_SUBJECT.begin(), POP_HEADER_SPAM_SUBJECT.end());
                        subject_processed_ = true;
                    }
                }
            }
            if (current_lines_ < 0 &&
                ((back_str.size() == 1 && back_str[0] == '\n') ||
                 (back_str.size() == 2 && back_str[0] == '\r' && back_str[1] == '\n')))
            {
                if (message.is_spam && popContext->settings.spamSubjectMarkEnable &&
                    !subject_processed_)
                {
                    current_list_->insert(
                        --current_list_->end(),
                        StringPtr(new string("Subject: " + POP_HEADER_SPAM_SUBJECT + "\r\n")));
                    subject_processed_ = true;
                }
                current_lines_ = 0;
                current_list_ = &body_;
            }
            if (current_lines_ < 0 &&
                (back_str.size() == 1 || back_str[back_str.size() - 2] != '\r'))
            {
                back_str[back_str.size() - 1] = '\r';
                back_str.push_back('\n');
            }
            current_list_->push_back(StringPtr(new string));
        }
        if (data == end)
        {
            current_list_->pop_back();
            return;
        }

        if (current_list_->back()->empty() && *data == '.') current_list_->back()->push_back('.');
        current_list_->back()->reserve(current_list_->back()->size() + i - data + 1);
        current_list_->back()->append(data, i);
    }

    void onMessageError(const string& str_err) override
    {
        error_ = str_err;
    }

    void onLog(const string& text) override
    {
        hasLog_ = true;
        logStr_ = text + ", mid=" + message.mid;
    }
    void onLogWarning(const string& text) override
    {
        hasLog_ = true;
        logStr_ = text + ", mid=" + message.mid;
    }

    const std::list<StringPtr>& headers() const
    {
        return headers_;
    }
    const std::list<StringPtr>& body() const
    {
        return body_;
    }
    const std::string& error() const
    {
        return error_;
    }

    const std::string& logStr() const
    {
        return logStr_;
    }
    bool hasLog()
    {
        return hasLog_;
    }

private:
    pop_context_ptr popContext;
    std::list<StringPtr> headers_;
    std::list<StringPtr> body_;
    int lines_;
    int current_lines_;
    bool receive_stopped_;
    bool subject_processed_;
    const message_entry& message;
    std::list<StringPtr>* current_list_;

    string error_;
    string logStr_;
    bool hasLog_;
};

void processor_impl::loadMessage(pop_args_ptr args, int lines)
{
    // Main function logic.
    message_entry* msg = get_msg_entry(args->messageIndex(), args->context);
    if (!msg)
    {
        return args->out_incorrect_msgid();
    }

    auto handler = std::make_shared<PopMessageHandler>(args->context, *msg, lines);
    try
    {
        if (msg->st_id.empty())
        {
            args->context->pop3Backend->extendMessage(*msg); // LOAD stid
        }

        if (!args->context->messages->checkStid(msg->st_id, settings_->mulcaMaxFailures))
        {
            timeoutError(
                args,
                "[SYS/TEMP]",
                "Unable to retrieve message, contact system administrator.",
                settings_->mulcaFailureTimout);
            return;
        }

        auto warningLog = [](const std::string& s) { YLOG_GLOBAL(warning) << s; };
        auto metaBackend = std::make_shared<PgMessageLoader>(*args->context, settings_);
        auto mulcaLoader = std::make_shared<yimap::mbody::MessageLoader>(
            msg->st_id,
            settings_->mbodyStorageOptions,
            args->context,
            std::move(warningLog),
            metaBackend->getMimes(msg->mid),
            0);

        auto future = mulcaLoader->load(yimap::mbody::LoadMessage, "", msg->db_size, handler);
        future.get();

        if (handler->headers().empty() && handler->body().empty())
        {
            args->context->messages->markBadStid(msg->st_id);
            throw std::runtime_error(std::string("mulca error: ") + handler->error());
        }

        if (handler->hasLog())
        {
            service_log_error(*args, handler->logStr());
        }
    }
    catch (const std::exception& e)
    {
        service_log_error(
            *args,
            std::string("can't load message: ") + e.what() + std::string(" mid=") + msg->mid);
        args->ostr->clientStream()
            << "-ERR [SYS/TEMP] unable to retrieve message, contact system administrator."
            << args->get_uniq_id_ending() << "\r\n";
        args->prom.set(ProcessorResult::FAILURE);
        return;
    }

    args->rememberResponse(args->messageIndex() + " " + msg->mid);

    args->ostr->clientStream() << "+OK " << msg->size << " octets.\r\n";
    args->ostr->clientStream() << POP_HEADER_FOLDER_NAME << msg->folder_name << "\r\n";
    bool add_finish_clrf = handler->body().empty();
    {
        std::list<StringPtr>::const_iterator i_next = handler->headers().begin();
        ++i_next;
        for (std::list<StringPtr>::const_iterator i = handler->headers().begin(),
                                                  i_end = handler->headers().end();
             i != i_end;
             ++i)
        {
            if (i_next == i_end)
            {
                if (msg->is_spam && args->context->settings.spamEnable)
                    args->ostr->clientStream() << POP_HEADER_SPAM_FLAG << "\r\n";
                if (lines == 0) break;
            }
            else
                ++i_next;
            args->ostr->clientStream() << **i;
        }
    }
    if (lines != 0)
    {
        for (std::list<StringPtr>::const_iterator i = handler->body().begin(),
                                                  i_end = handler->body().end();
             i != i_end;
             ++i)
        {
            args->ostr->clientStream() << **i;
            add_finish_clrf = ((*i)->size() == 1 || *((*i)->end() - 2) != '\r');
        }
    }
    if (add_finish_clrf) args->ostr->clientStream() << "\r\n.\r\n";
    else
        args->ostr->clientStream() << ".\r\n";

    args->context->retr_messages++;

    wmi_markread_message(args->context, msg);
    args->prom.set(ProcessorResult::SUCCESS);
}

void processor_impl::cmd_top(pop_args_ptr args)
{
    int lines = -1;
    if (args->req->size() != 2 && args->req->size() != 3)
        return args->out_incorrect_command("top <msg-id> [lines]");

    if (args->req->size() == 3)
    {
        try
        {
            lines = boost::lexical_cast<int>((*args->req)[2]);
        }
        catch (boost::bad_lexical_cast&)
        {
            return args->out_incorrect_command("top <msg-id> [lines]");
        }
    }

    loadMessage(args, lines);
}

void processor_impl::cmd_retr(pop_args_ptr args)
{
    if (args->req->size() != 2) return args->out_incorrect_command("retr <msg-id>");

    loadMessage(args, -1);
}

void processor_impl::cmd_dele(pop_args_ptr args)
{
    if (args->req->size() != 2) return args->out_incorrect_command("quit <msg-id>");

    int msg_id = get_msg_index(args->messageIndex(), args->context);
    if (msg_id < 0) return args->out_incorrect_msgid();

    args->context->messages->markDeleted(msg_id);
    args->rememberResponse(args->messageIndex() + " " + args->context->messages->at(msg_id).mid);
    args->out_statistic();
}

void processor_impl::cmd_rset(pop_args_ptr args)
{
    if (args->req->size() > 1) return args->out_incorrect_command("rset");

    args->context->messages->resetDeleted();
    args->out_statistic();
}

void processor_impl::cmd_quit(pop_args_ptr args)
{
    if (args->req->size() > 1) return args->out_incorrect_command("quit");

    if (args->context->messages->hasDeleted())
    {
        deleteMessages(args);
    }
    args->out_goobye();
}

void processor_impl::deleteMessages(pop_args_ptr args)
{
    if (!args->context->messages) return;

    std::list<std::string> mids;
    for (const auto& message : *args->context->messages)
    {
        if (message.is_deleted) mids.push_back(message.mid);
    }

    if (mids.empty()) return;

    auto messageLoader = std::make_shared<PgMessageLoader>(*args->context, settings_);
    messageLoader->deleteMessages(mids);

    namespace id = user_journal::parameters::id;
    const std::string midsStr = boost::algorithm::join(mids, ",");
    std::vector<string> midsVec(mids.begin(), mids.end());
    journalOperation<user_journal::parameters::DeleteMessages>(
        args->context, id::state(midsStr), id::affected(mids.size()), id::mids(midsVec));
}

// If given msg message is new and we should mark new messages as read,
// then add its mid to stack. If this stack has enough messages - sent request to wmi.
void processor_impl::wmi_markread_message(pop_context_ptr context, ypop::message_entry* msg)
{
    // If we should not mark messages as read, or if current message is not new - return.
    if (!context->settings.markRead || !msg->is_new)
    {
        return;
    }

    // Clear message is_new flag.
    msg->is_new = false;

    // Mark as read batch of mail ids. Batch size is in config.
    LOCK_POP_CONTEXT(*context);
    context->mark_read_messages.push_back(msg->mid);
    context->mark_read_message_hook = boost::bind(&processor_impl::markRead, this, context);

    if (context->mark_read_messages.size() < settings_->mark_read_batch_size)
    {
        return;
    }

    UNLOCK_POP_CONTEXT();

    markRead(context);
}

// Returns true if mark_read request was sent to WMI.
// If request was delayed, returns false.
void processor_impl::markRead(pop_context_ptr context)
{
    std::list<std::string> mids;
    // Mark as read batch of mail ids. Batch size is in config.
    {
        LOCK_POP_CONTEXT(*context);
        if (context->mark_read_messages.empty())
        {
            return;
        }

        for (; !context->mark_read_messages.empty(); context->mark_read_messages.pop_back())
        {
            string next_mid = context->mark_read_messages.back();
            mids.push_back(next_mid);
        }
    }

    auto messageLoader = std::make_shared<PgMessageLoader>(*context, settings_);
    messageLoader->markReadMessages(mids);
}

} // namespace ypop
