#ifndef _YPOP_PROCESSOR_MESSAGE_LIST_H_
#define _YPOP_PROCESSOR_MESSAGE_LIST_H_

#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/locks.hpp>

#include <string>
#include <vector>
#include <deque>

typedef boost::mutex mutex_t;
typedef boost::unique_lock<mutex_t> lock_t;

namespace ypop {

using namespace std;

struct message_entry
{
    std::string mid;
    std::string st_id;
    std::string uidl;
    std::string folder_name;

    std::size_t size = 0;
    std::size_t db_size = 0;
    bool is_new = false;
    bool is_spam = false;
    bool is_deleted = false;
};

using MessageVector = std::vector<message_entry>;

class MessageList : public MessageVector
{
public:
    MessageList(const string& folderId) : folderId(folderId)
    {
    }

    bool singleFolder() const
    {
        return !folderId.empty();
    }

    bool hasDeleted() const
    {
        return deleted > 0;
    }

    void cashData()
    {
        MessageVector::const_iterator i = begin();
        MessageVector::const_iterator iend = end();

        deleted = 0;
        good_size = 0;
        good_number = 0;
        for (; i != iend; ++i)
        {
            if (!i->is_deleted)
            {
                good_size += i->size;
                ++good_number;
            }
            else
            {
                deleted++;
            }
        }
    }

    void resetDeleted()
    {
        for (MessageVector::iterator i = begin(), iend = end(); i != iend; ++i)
        {
            i->is_deleted = false;
        }
        cashData();
    }

    void markDeleted(int id)
    {
        message_entry& message = (*this)[id];
        if (message.is_deleted) return;

        message.is_deleted = true;
        good_size -= message.size;
        good_number--;
        deleted++;
    }

    bool checkStid(const string& stid, size_t maxFailures)
    {
        lock_t lock(mux);
        std::map<string, size_t>::const_iterator found = badStids.find(stid);
        if (found == badStids.end()) return true;
        return found->second < maxFailures;
    }

    void markBadStid(const string& stid)
    {
        lock_t lock(mux);
        std::map<string, size_t>::iterator found = badStids.find(stid);
        size_t count = found == badStids.end() ? 0 : found->second;
        badStids[stid] = count + 1;
    }

public:
    int deleted = 0;
    size_t good_size = 0;
    size_t good_number = 0;
    const string folderId;

    std::map<string, size_t> badStids;
    mutable mutex_t mux;
};

typedef boost::shared_ptr<MessageList> MessageListPtr;

} // namespace ypop

#endif // _YPOP_PROCESSOR_MESSAGE_LIST_H_
