#ifndef _YPOP_PROCESSOR_MODULE_CONTEXT_H_
#define _YPOP_PROCESSOR_MODULE_CONTEXT_H_

#include <processor/config.h>
#include <yplatform/module.h>
#include <yplatform/util/unordered.h>

namespace ypop {

struct execution
{
    size_t calls = 0;
    yplatform::time_traits::duration duration = yplatform::time_traits::duration::zero();
    size_t status_success = 0;
    size_t status_fail = 0;
    size_t status_uncertain = 0;
};

class processor_stats : public yplatform::module_stats
{
public:
    typedef yplatform::iunordered<string, execution>::map stat_map_t;

    processor_stats()
        : cmds(0)
        , backend_reqs(0)
        , logins_granted(0)
        , logins_rejected(0)
        , logins_bb_errors(0)
        , logins_api_errors(0)
        , logins_disabled(0)
    {
    }

    size_t cmds;
    size_t backend_reqs;
    size_t logins_granted;
    size_t logins_rejected;
    size_t logins_bb_errors;
    size_t logins_api_errors;
    size_t logins_disabled;
    stat_map_t commands;
    stat_map_t api_calls;

protected:
    virtual ptree_ptr core_get_stat() const
    {
        ptree_ptr result = yplatform::module_stats::core_get_stat();
        result->put("cmds", cmds);
        result->put("backend_requests", backend_reqs);
        result->put("logins.granted", logins_granted);
        result->put("logins.rejected", logins_rejected);
        result->put("logins.bb_errors", logins_bb_errors);
        result->put("logins.api_errors", logins_api_errors);
        result->put("logins.disabled", logins_disabled);
        for (stat_map_t::const_iterator i = commands.begin(), i_end = commands.end(); i != i_end;
             ++i)
        {
            string path = "commands." + i->first + ".";
            result->put(path + "calls", i->second.calls);
            result->put(
                path + "duration",
                yplatform::time_traits::to_string(
                    yplatform::time_traits::duration_cast<yplatform::time_traits::seconds>(
                        i->second.duration)));
            result->put(path + "status_success", i->second.status_success);
            result->put(path + "status_fail", i->second.status_fail);
            result->put(path + "status_fail", i->second.status_uncertain);
        }
        for (stat_map_t::const_iterator i = api_calls.begin(), i_end = api_calls.end(); i != i_end;
             ++i)
        {
            string path = "api_calls." + i->first + ".";
            result->put(path + "calls", i->second.calls);
            result->put(
                path + "duration",
                yplatform::time_traits::to_string(
                    yplatform::time_traits::duration_cast<yplatform::time_traits::seconds>(
                        i->second.duration)));
            result->put(path + "status_success", i->second.status_success);
            result->put(path + "status_fail", i->second.status_fail);
            result->put(path + "status_fail", i->second.status_uncertain);
        }
        return result;
    }
};

typedef boost::shared_ptr<processor_stats> processor_stats_ptr;

class task_context : public yplatform::module_task_context
{
public:
    enum login_status
    {
        granted,
        rejected,
        bb_error,
        api_error,
        disabled,
        unknown
    };
    enum exec_status
    {
        success,
        fail,
        uncertain
    };

    task_context() : cmds_(0), backend_reqs_(0)
    {
    }

    void set_module_stat(processor_stats_ptr module_stat)
    {
        module_stat_ = module_stat;
    }

    void stats(unsigned long cmds, unsigned long reqs, string const& cmd = string())
    {
        lock_t lock(mux());
        cmds_ += cmds;
        backend_reqs_ += reqs;
        if (!cmd.empty())
        {
            cmd_last_ = cmd;
            cmd_time_ = ::yplatform::time_traits::clock::now();
        }
        {
            boost::unique_lock<boost::mutex> lck(module_stat_->mux());
            module_stat_->cmds += cmds;
            module_stat_->backend_reqs += reqs;
        }
    }

    void stats_login(login_status status)
    {
        boost::unique_lock<boost::mutex> lck(module_stat_->mux());
        switch (status)
        {
        case granted:
            ++module_stat_->logins_granted;
            break;
        case rejected:
            ++module_stat_->logins_rejected;
            break;
        case bb_error:
            ++module_stat_->logins_bb_errors;
            break;
        case api_error:
            ++module_stat_->logins_api_errors;
            break;
        case disabled:
            ++module_stat_->logins_disabled;
            break;
        default:
            break;
        }
    }

    void stats_command(
        string const& cmd,
        yplatform::time_traits::duration const& dur,
        exec_status status)
    {
        boost::unique_lock<boost::mutex> lck(module_stat_->mux());
        execution& ex = module_stat_->commands[cmd];

        ++ex.calls;
        ex.duration += dur;

        switch (status)
        {
        case success:
            ++ex.status_success;
            break;
        case fail:
            ++ex.status_fail;
            break;
        case uncertain:
            ++ex.status_uncertain;
            break;
        }
    }

    void stats_api(
        string const& name,
        yplatform::time_traits::duration const& dur,
        exec_status status)
    {
        boost::unique_lock<boost::mutex> lck(module_stat_->mux());
        execution& ex = module_stat_->api_calls[name];

        ++ex.calls;
        ex.duration += dur;

        switch (status)
        {
        case success:
            ++ex.status_success;
            break;
        case fail:
            ++ex.status_fail;
            break;
        case uncertain:
            ++ex.status_uncertain;
            break;
        }
    }

protected:
    virtual ptree_ptr core_get_stat() const
    {
        ptree_ptr result = yplatform::module_task_context::core_get_stat();
        result->put("commands", cmds_);
        result->put("backend_requests", backend_reqs_);
        result->put("last_command", cmd_last_);
        result->put("last_command_start", cmd_time_.time_since_epoch());
        return result;
    }

private:
    unsigned long cmds_;
    unsigned long backend_reqs_;
    string cmd_last_;
    yplatform::time_traits::time_point cmd_time_;
    processor_stats_ptr module_stat_;
};

typedef boost::shared_ptr<task_context> task_context_ptr;

}

#endif // _YPOP_PROCESSOR_MODULE_CONTEXT_H_
