#pragma once

#include <processor/command.h>
#include <processor/settings.h>
#include <user_settings.h>
#include <processor/message.h>
#include <processor/message_list.h>

#include <macs/connection_info.h>
#include <macs/mime_part.h>

#include <backend/backend_types.h>

namespace ypop {

using PromiseMessageList = yplatform::future::promise<MessageListPtr>;
using FutureMessageList = yplatform::future::future<MessageListPtr>;

class MessageLoader : public std::enable_shared_from_this<MessageLoader>
{
public:
    MessageLoader(pop_context& context, PopSettingsPtr settings);
    virtual ~MessageLoader()
    {
    }

    virtual FutureMessageList loadUidl() = 0;
    virtual void extendMessage(message_entry& msg) = 0;

    MessageListPtr getMessages();

    virtual macs::MimeParts getMimes(const std::string& mid) = 0;
    virtual void markReadMessages(const std::list<std::string>& mids) = 0;
    virtual void deleteMessages(const std::list<std::string>& mids) = 0;

protected:
    virtual macs::EnvelopesRepository& envelopes() = 0;
    virtual macs::LabelsRepository& labels() = 0;
    virtual macs::FoldersRepository& folders() = 0;
    virtual macs::ImapRepository& imapRepo() = 0;

    size_t getMessageExtraSize(const ypop::message_entry& ind);

    void postprocessMessages();
    void messageFolder(const string& srcName, message_entry& message);

    macs::Label getSeenLabel();

protected:
    MessageLoader(const MessageLoader&) = delete;
    enum
    {
        POP_SORT_THRESHOLD = 1000
    };

    pop_context& popContext;
    PopSettingsPtr settings_;

    MessageListPtr messages;
    boost::optional<macs::Label> seenLabel;

    std::map<string, string> folder_cash_;
    std::string exception;
};

class PgMessageLoader : public MessageLoader
{
public:
    PgMessageLoader(pop_context& context, PopSettingsPtr settings);
    PgMessageLoader(const PgMessageLoader&) = delete;

    virtual FutureMessageList loadUidl() override;

    virtual void extendMessage(message_entry& msg) override;

    virtual macs::MimeParts getMimes(const std::string& mid) override;
    virtual void markReadMessages(const std::list<std::string>& mids) override;
    virtual void deleteMessages(const std::list<std::string>& mids) override;

protected:
    macs::Service& master();
    macs::Service& replica();

    macs::EnvelopesRepository& envelopes() override;
    macs::LabelsRepository& labels() override;
    macs::FoldersRepository& folders() override;
    macs::ImapRepository& imapRepo() override;

    yimap::backend::PgServicePtr pgMaster;
    yimap::backend::PgServicePtr pgReplica;
};

} // namespace ypop
