#include <processor/pop_backend.h>
#include <yplatform/util/sstream.h>
#include <backend/backend_session.h>
#include <macs_pg/macs_pg.h>

namespace ypop {

//-----------------------------------------------------------------------------
// PG POP3 backend implementaion

PgMessageLoader::PgMessageLoader(pop_context& context, PopSettingsPtr settings)
    : MessageLoader(context, settings)
{
}

FutureMessageList PgMessageLoader::loadUidl()
{
    std::string spamFid;
    std::list<std::string> fids;
    auto prom = std::make_shared<PromiseMessageList>();

    try
    {
        spamFid = folders().getFolderFidBySymbol(macs::Folder::Symbol::spam);
        auto allFolders = folders().getAllFolders();
        std::string defaultFid;
        for (const auto& folder : allFolders)
        {
            if (folder.second.symbolicName() == macs::Folder::Symbol::inbox)
            {
                defaultFid = folder.second.fid();
            }
            if (folder.second.pop3On())
            {
                fids.push_back(folder.second.fid());
            }
        }

        if (fids.empty())
        {
            fids.push_back(defaultFid);
        }

        messages.reset(new MessageList(""));
    }
    catch (const std::exception& e)
    {
        prom->set_exception(e);
        return *prom;
    }

    auto self = shared_from_this();
    replica().imapRepo().pop3LoadFullMessages(
        popContext.suid,
        spamFid,
        fids,
        [this, self, spamFid, prom](
            const macs::sys::error_code& ec, const macs::Pop3MessageChunk& messageChunk) {
            if (ec)
            {
                prom->set_exception(std::runtime_error(ec.message()));
                return;
            }
            for (auto& message : messageChunk)
            {
                message_entry newMessage;
                newMessage.mid = message.mid;
                newMessage.uidl = message.base.uidl.empty() ? message.mid : message.base.uidl;
                newMessage.size = message.base.size;
                newMessage.db_size = message.base.size;
                newMessage.folder_name = message.base.fid;
                newMessage.is_spam = message.base.fid == spamFid;

                messages->push_back(newMessage);
            }

            postprocessMessages();
            prom->set(messages);
        });
    return *prom;
}

void PgMessageLoader::extendMessage(message_entry& msg)
{
    if (msg.st_id.empty())
    {
        auto message = replica().envelopes().getById(msg.mid);
        msg.st_id = message.stid();
        msg.is_new = !message.hasLabel(getSeenLabel());
    }
}

macs::MimeParts PgMessageLoader::getMimes(const std::string& mid)
{
    try
    {
        macs::MimeParts mimes;
        macs::Mids mids = { mid };
        auto mimesWithMids = envelopes().getMimesWithDeleted(mids);
        if (!mimesWithMids.empty())
        {
            std::tie(std::ignore, std::ignore, mimes) = std::move(mimesWithMids.front());
        }
        else
        {
            throw std::runtime_error("no such message, mid=" + mid);
        }
        return mimes;
    }
    catch (const std::exception& e)
    {
        throw std::runtime_error(std::string("can't get mimes: ") + e.what());
    }
}

void PgMessageLoader::markReadMessages(const std::list<std::string>& mids)
{
    try
    {
        std::list<macs::Label> labels;
        labels.push_back(getSeenLabel());
        master().envelopes().markEnvelopes(mids, labels);
    }
    catch (...)
    {
        TASK_LOG(&popContext, error)
            << popContext.to_log_string() << "db error: failed to mark messages as read";
    }
}
void PgMessageLoader::deleteMessages(const std::list<std::string>& mids)
{
    yplatform::future::promise<macs::Revision> promiseDelete;
    master().imapRepo().pop3DeleteMessages(
        mids, [&promiseDelete](macs::sys::error_code const& ec, macs::Revision rev) {
            if (ec)
            {
                promiseDelete.set_exception(std::runtime_error(ec.message()));
            }
            else
            {
                promiseDelete.set(rev);
            }
        });

    yplatform::future::future<macs::Revision> futureDelete(promiseDelete);
    try
    {
        futureDelete.get();
    }
    catch (const std::exception& e)
    {
        TASK_LOG(&popContext, error)
            << popContext.to_log_string() << " db error: failed to delete pop3 messages, mdb=pg"
            << ", exception='" << e.what() << '\'';
    }
}

macs::Service& PgMessageLoader::master()
{
    if (!pgMaster)
    {
        auto backendService = yplatform::find<yimap::backend::BackendService>("mod_macs");
        pgMaster = backendService->createPgService(
            yimap::createShortSessionInfo(popContext), false, "yserver_pop");
    }
    return *pgMaster;
}
macs::Service& PgMessageLoader::replica()
{
    if (!pgReplica)
    {
        auto backendService = yplatform::find<yimap::backend::BackendService>("mod_macs");
        pgReplica = backendService->createPgService(
            yimap::createShortSessionInfo(popContext), true, "yserver_pop");
    }
    return *pgReplica;
}

macs::EnvelopesRepository& PgMessageLoader::envelopes()
{
    return replica().envelopes();
}

macs::LabelsRepository& PgMessageLoader::labels()
{
    return replica().labels();
}

macs::FoldersRepository& PgMessageLoader::folders()
{
    return replica().folders();
}

macs::ImapRepository& PgMessageLoader::imapRepo()
{
    return master().imapRepo();
}

} // namespace ypop
