#pragma once

#include <common/context.h>
#include <user_settings.h>
#include "message_list.h"

#include <yplatform/net/types.h>
#include <boost/thread.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/crc.hpp>

#include <ctime>
#include <string>
#include <vector>
#include <deque>

namespace ypop {

using namespace std;

using Timer = yplatform::time_traits::timer;
using TimerPtr = yplatform::net::timer_ptr;

typedef boost::function<void()> void_hook_t;
typedef boost::function<TimerPtr(uint32_t, void_hook_t)> CreateTimerHook;

typedef std::vector<std::string> command_t;
typedef boost::shared_ptr<command_t> request_t;

// Message loading backend
class MessageLoader;

struct pop_context : public yimap::Context
{
    enum PopSessionState
    {
        START,
        NAME,
        TRANS,
        QUIT,
        ANY,
        INVALID
    };

    pop_context() : state(START), messages(new MessageList(""))
    {
    }

    ~pop_context()
    {
        std::ostringstream stream;
        size_t cnt = 0;
        for (size_t i = 0; i < command_history.size(); i++)
        {
            stream << "\"" << command_history[i] << "\"";

            if (i != command_history.size() - 1 && cnt != BLOCK_SIZE_COMMAND_HISTOTY - 1)
            {
                stream << ",";
            }

            if (cnt == BLOCK_SIZE_COMMAND_HISTOTY - 1)
            {
                L_(info) << uniq_id() << ": command history: " << stream.str();
                stream.str("");
                stream.clear();
                cnt = 0;
            }
            else
            {
                ++cnt;
            }
        }

        if (!stream.str().empty())
        {
            L_(info) << uniq_id() << ": command history: " << stream.str();
        }
    }

    virtual const std::string& get_name() const
    {
        static const std::string NAME = "pop_session";
        return NAME;
    }

    TimerPtr activeTimer;

    PopSessionState state = START;
    MessageListPtr messages;

    std::deque<std::string> command_history;

    // TLS settings
    bool tls_on = false;
    string sslStatus = "no";

    UserSettings settings;

    bool list_requested = false;
    std::size_t msize_requested = 0;
    std::size_t muidl_requested = 0;
    // Number of RETR calls during session.
    std::size_t retr_messages = 0;

    string command;

    // List of messages, that should be marked as seen in WMI.
    std::deque<std::string> mark_read_messages;
    // If we have some messages in queue to mark as read, this hook do it after session is closed.
    boost::function<void()> mark_read_message_hook;

    boost::crc_32_type::value_type passwordCrc;

    std::string state_name() const;
    std::string to_log_string() const;

    bool checkStid(const string& stid, size_t maxFailures);
    void markBadStid(const string& stid);

    void addHistoryEntry(std::string&& entry);

    std::shared_ptr<MessageLoader> pop3Backend;

protected:
    virtual ptree_ptr core_get_stat() const
    {
        ptree_ptr result = yplatform::task_context::core_get_stat();
        result->put("state", state_name());
        result->put("auth.login", login);
        result->put(
            "local_addr",
            sessionInfo.local_address + ":" +
                boost::lexical_cast<std::string>(sessionInfo.local_port));
        result->put(
            "remote_addr",
            sessionInfo.remote_address + ":" +
                boost::lexical_cast<std::string>(sessionInfo.remote_port));
        result->put("ssl", tls_on);
        result->put("auth.suid", suid);
        result->put("auth.uid", uid);
        result->put("auth.storage", storage);
        return result;
    }
    enum
    {
        BLOCK_SIZE_COMMAND_HISTOTY = 10
    };
};

typedef boost::shared_ptr<pop_context> pop_context_ptr;

#define LOCK_POP_CONTEXT(ctx) ypop::pop_context::lock_t lock((ctx).mux())
#define UNLOCK_POP_CONTEXT() lock.unlock()

} // namespace ypop
