#include <processor/processor_impl.h>
#include <network/session_fwd.h>
#include <backend/backend_session.h>
#include <syslog.h>

namespace ypop {

using yimap::server::StartTls;
using yimap::server::ResumeReading;
using yimap::server::ShutdownSession;

void processor_impl::deadline_cancel_hook(pop_args_ptr args)
{
    args->context->state = session_context_t::QUIT;
    args->out_server_error();
}

processor_impl::processor_impl()
    : Processor(), stats_(new processor_stats), enqueue_timeout_(boost::posix_time::pos_infin)
{
    L_(info) << "pop_processor_impl task instaniated";
}

processor_impl::~processor_impl()
{
    L_(info) << "pop_processor_impl task destroyed";
}

void processor_impl::init(const yplatform::ptree& xml)
{
    settings_.reset(new PopSettings(xml));

    commands_["noop"] = cmd_entry(&processor_impl::cmd_noop, session_context_t::ANY);
    commands_["capa"] = cmd_entry(&processor_impl::cmd_capa, session_context_t::ANY);
    commands_["user"] = cmd_entry(&processor_impl::cmd_user, session_context_t::START);
    commands_["pass"] = cmd_entry(&processor_impl::cmd_pass, session_context_t::NAME);
    commands_["stls"] = cmd_entry(&processor_impl::cmd_stls, session_context_t::START);
    commands_["stat"] = cmd_entry(&processor_impl::cmd_stat, session_context_t::TRANS);
    commands_["vers"] = cmd_entry(&processor_impl::cmd_vers, session_context_t::ANY);
    commands_["list"] = cmd_entry(&processor_impl::cmd_list, session_context_t::TRANS);
    commands_["uidl"] = cmd_entry(&processor_impl::cmd_uidl, session_context_t::TRANS);
    commands_["dele"] = cmd_entry(&processor_impl::cmd_dele, session_context_t::TRANS);
    commands_["quit"] = cmd_entry(&processor_impl::cmd_quit, session_context_t::ANY);
    commands_["top"] = cmd_entry(&processor_impl::cmd_top, session_context_t::TRANS);
    commands_["retr"] = cmd_entry(&processor_impl::cmd_retr, session_context_t::TRANS);
    commands_["rset"] = cmd_entry(&processor_impl::cmd_rset, session_context_t::TRANS);
}

void processor_impl::fini(void)
{
}

ypop::message_entry* processor_impl::get_msg_entry(const string& msg_id, pop_context_ptr context)
    const
{
    std::size_t id = 0;
    try
    {
        id = boost::lexical_cast<std::size_t>(msg_id);
        --id;
    }
    catch (boost::bad_lexical_cast&)
    {
        return 0;
    }

    if (id >= context->messages->size()) return 0;

    if ((*context->messages)[id].is_deleted) return 0;

    return &(*context->messages)[id];
}

int processor_impl::get_msg_index(const string& msg_id, pop_context_ptr context) const
{
    int id = 0;
    id = ::atoi(msg_id.c_str()) - 1;

    if (id >= static_cast<int>(context->messages->size()) || id < 0) return -1;

    if ((*context->messages)[id].is_deleted) return -1;

    return id;
}

void processor_impl::process(
    const request_t& req,
    pop_context_ptr context,
    NetworkSessionPtr ostr,
    const yplatform::active::ptime&)
{
    if (req->empty()) return sendBadCommand(ostr, context, req);

    auto icmd = commands_.find((*req)[0]);
    if (icmd == commands_.end()) return sendBadCommand(ostr, context, req);

    if (icmd->second.state != context->state && icmd->second.state != session_context_t::ANY)
    {
        return sendBadCommand(ostr, context, req);
    }

    auto command = createCommand(req, context, ostr);
    ostr->postOnStrand(
        std::bind(&processor_impl::process_i, get_shared_from_this(), command, icmd->second));

    yplatform::future::future<response> future = command->prom;
    future.add_callback(
        boost::bind(&processor_impl::onCommandExecuted, get_shared_from_this(), command));
}

pop_args_ptr processor_impl::createCommand(
    const request_t& req,
    pop_context_ptr context,
    NetworkSessionPtr ostr)
{
    return std::make_shared<PopCommand>(req, context, ostr, stats_);
}

void processor_impl::sendBadCommand(
    NetworkSessionPtr ostr,
    pop_context_ptr context,
    const request_t& req)
{
    context->addHistoryEntry(boost::join(*req, " "));
    context->addHistoryEntry((req->empty() ? "" : req->at(0)) + " <-ERR What?>");

    ostr->clientStream() << "-ERR What?" << yimap::createErrorSuffix(context->uniq_id()) << "\r\n";
    ostr->postEvent(std::make_shared<ResumeReading>());
}

void processor_impl::process_i(pop_args_ptr command, cmd_entry cmdEntry)
{
    assert(!command->req->empty());
    command->statistic->stats(1, 0, (*command->req)[0]);
    command->remember();
    (this->*(cmdEntry.cmd_ptr))(command);
}

void processor_impl::timeoutError(
    pop_args_ptr args,
    const string& code,
    const string& response,
    int timeout)
{
    LOCK_POP_CONTEXT(*args->context);

    if (timeout <= 0)
    {
        outError(args, code, response);
    }
    else
    {
        auto hook = std::bind(
            &processor_impl::outError,
            get_shared_from_this(),
            args,
            string(code),
            string(response));
        args->context->activeTimer = args->ostr->createTimer(static_cast<uint32_t>(timeout), hook);
    }
}

void processor_impl::outError(pop_args_ptr args, string code, string response)
{
    args->ostr->clientStream() << "-ERR " << code << " " << response << " "
                               << args->get_uniq_id_ending() << "\r\n";
    args->prom.set(ProcessorResult::FAILURE);
    if (args->context->activeTimer) args->context->activeTimer.reset();
}

void processor_impl::logCommand(pop_args_ptr args)
{
    yplatform::future::future<response> response = args->prom;
    std::string status;
    try
    {
        status = toString(response.get().result_code);
    }
    catch (...)
    {
        status = "exception";
    }
    using yplatform::log::typed::make_attr;
    auto& ctx = args->context;
    YLOG(custom_loggers_.typed_log(), info)
        << make_attr("y_context", ctx->uniq_id()) << make_attr("uid", ctx->uid)
        << make_attr("suid", ctx->suid) << make_attr("type", "command_finished")
        << make_attr("command", args->command) << make_attr("status", status);
}

void processor_impl::onCommandExecuted(pop_args_ptr args)
{
    logCommand(args);

    yplatform::future::future<response> response = args->prom;
    if (response.has_exception())
    {
        string reason = "unknown";
        try
        {
            response.get();
        }
        catch (std::exception& e)
        {
            reason = e.what();
        }
        catch (...)
        {
        }
        args->ostr->getLogger().logError()
            << "processor return exception. reason='" << reason << "'";
        args->ostr->postEvent(std::make_shared<ShutdownSession>());
        return;
    }

    if (response.get().result_code == ProcessorResult::START_TLS)
    {
        // In fact, TLS starts further, in Pop3Session::handle_write()
        args->ostr->postEvent(std::make_shared<StartTls>());
        args->context->sslStatus = "starttls";
        return;
    }

    if (response.get().result_code == ProcessorResult::FAILURE_CLOSE)
    {
        args->ostr->postEvent(std::make_shared<ShutdownSession>());
        return;
    }

    args->ostr->postEvent(std::make_shared<ResumeReading>());
}

DEFINE_SERVICE_OBJECT(ypop::processor_impl)

} // namespace ypop
