#ifndef _YPOP_RPOCESSOR_IMPL_H_
#define _YPOP_RPOCESSOR_IMPL_H_

#include <boost/thread.hpp>
#include <boost/variant.hpp>
#include <boost/property_tree/ptree.hpp>
#include <boost/shared_ptr.hpp>

#include <yplatform/exception.h>
#include <yplatform/repository.h>
#include <yplatform/find.h>
#include <yplatform/log.h>

#include <backend/backend_session.h>

#include <server/server.h>

#include <processor/module_context.h>
#include <processor/logger.h>
#include <processor/command.h>
#include <processor/settings.h>
#include <user_settings.h>

#include <ymod_blackbox/auth.h>

#include <macs/hooks.h>
#include <macs/user_journal.h>

#include <map>

namespace ypop {

class MessageLoader;

typedef boost::shared_ptr<boost::property_tree::ptree> ptree_ptr;
typedef yplatform::future::future<ptree_ptr> future_ptree_ptr;
typedef yplatform::future::future<void> future_void;

using BlackboxPromise = yplatform::future::future<ymod_blackbox::response>;
using FutureMessageList = yplatform::future::future<MessageListPtr>;

typedef boost::optional<yplatform::future::promise<response>> optional_promise;

class processor_impl : public Processor
{
private:
    virtual const yplatform::active::time_duration& get_enqueue_timeout() const
    {
        return enqueue_timeout_;
    }

public:
    processor_impl();
    virtual ~processor_impl();

    void init(const yplatform::ptree& xml);
    void fini(void);

protected:
    template <typename OperationType, typename... ArgsT>
    void journalOperation(pop_context_ptr context, ArgsT&&... args)
    {
        auto backendService = yplatform::find<yimap::backend::BackendService>("mod_macs");
        auto journal = backendService->createJournal(
            context->uid,
            context->sessionInfo.remote_address,
            context->storage,
            context->suid,
            "yserver_pop");

        if (journal)
        {
            try
            {
                journal->logOperation<OperationType>(std::forward<ArgsT>(args)...);
            }
            catch (const std::exception& e)
            {
                service_log_error_context(context, "user_journal error: " << e.what());
            }
            catch (...)
            {
                service_log_error_context(context, "user_journal error: unknown error");
            }
        }
    }

    void logCommand(pop_args_ptr args);

    std::shared_ptr<const processor_impl> get_shared_from_this() const
    {
        return static_pointer_cast<const processor_impl>(shared_from_this());
    }

    std::shared_ptr<processor_impl> get_shared_from_this()
    {
        return static_pointer_cast<processor_impl>(shared_from_this());
    }

    std::string translate_request(const string& str) const;

    void deadline_cancel_hook(pop_args_ptr args);

    pop_args_ptr createCommand(
        const request_t& req,
        pop_context_ptr context,
        NetworkSessionPtr ostr);
    void process_i(pop_args_ptr command, cmd_entry cmdEntry);

    void sendBadCommand(NetworkSessionPtr ostr, pop_context_ptr context, const request_t& req);

    ypop::message_entry* get_msg_entry(const string& msg_id, pop_context_ptr context) const;

    int get_msg_index(const string& msg_id, pop_context_ptr context) const;

    // commands block

    void cmd_noop(pop_args_ptr args);
    void cmd_capa(pop_args_ptr args);
    void cmd_user(pop_args_ptr args);
    void cmd_pass(pop_args_ptr args);
    void cmd_stls(pop_args_ptr args);
    void cmd_stat(pop_args_ptr args);
    void cmd_vers(pop_args_ptr args);
    void cmd_list(pop_args_ptr args);
    void cmd_uidl(pop_args_ptr args);
    void cmd_dele(pop_args_ptr args);
    void cmd_quit(pop_args_ptr args);
    void cmd_top(pop_args_ptr args);
    void cmd_retr(pop_args_ptr args);
    void cmd_rset(pop_args_ptr args);
    // **************

    void blackbox_cb(pop_args_ptr args, BlackboxPromise var);
    bool checkAuthError(
        pop_args_ptr args,
        BlackboxPromise& authPromise,
        ymod_blackbox::response& result,
        std::string& errorPrivate,
        std::string& errorPublic);

    bool checkSSL(pop_args_ptr args);

    void deleteMessages(pop_args_ptr args);

    void loadSettingsAsync(pop_args_ptr args);
    void loadSettings(pop_args_ptr args);
    void settingsCallback(
        pop_args_ptr args,
        const boost::system::error_code& ec,
        const UserSettings& settings);
    void processSettings(pop_args_ptr args, const UserSettings& settings);

    void loadMessages(pop_args_ptr args);
    void handleMessages(pop_args_ptr args, FutureMessageList messages);

    void completeLogin(pop_args_ptr args);

    void loadMessage(pop_args_ptr args, int lines);
    // If given msg message is new and we should mark new messages as read,
    // then add its mid to stack. If this stack has enough messages - sent request to wmi.
    void wmi_markread_message(pop_context_ptr context, ypop::message_entry* msg);
    void markRead(pop_context_ptr context);

    void timeoutError(pop_args_ptr args, const string& code, const string& response, int timeout);
    void outError(pop_args_ptr args, string code, string response);

public:
    virtual void process(
        const request_t& req,
        pop_context_ptr context,
        NetworkSessionPtr ostr,
        const yplatform::active::ptime& deadline);

    virtual const yplatform::module_stats_ptr get_module_stats() const
    {
        return boost::static_pointer_cast<yplatform::module_stats>(stats_);
    }

protected:
    void onCommandExecuted(pop_args_ptr args);

    const custom_loggers& proc_loggers() const
    {
        return custom_loggers_;
    }

    custom_loggers custom_loggers_;
    processor_stats_ptr stats_;
    yplatform::active::time_duration enqueue_timeout_;
    std::map<string, cmd_entry> commands_;
    PopSettingsPtr settings_;
};

template <typename FutureT>
std::string get_exception_reason(const FutureT& f)
{
    try
    {
        f.get();
    }
    catch (::yplatform::exception& e)
    {
        return e.public_message();
    }
    catch (std::exception& e)
    {
        return e.what();
    }
    catch (...)
    {
        return "unknown";
    }
    return "no errors";
}

typedef void (processor_impl::*cmd_processor_t)(pop_args_ptr args);
} // namespace ypop

#endif // _YPOP_RPOCESSOR_IMPL_H_
