#include <server/pop3_session.h>
#include <processor/processor.h>
#include <server/command_parser.h>

#include <yplatform/log.h>
#include <yplatform/context_repository.h>

namespace ypop {

using yimap::server::StartTls;
using yimap::server::StartReader;
using yimap::server::ShutdownSession;
using yimap::server::ResumeReading;

static const std::string SERVER_IDENTIFIER = yimap::HostId::create().suffix;

Pop3Session::Pop3Session(const ServerSettings& settings, SessionPtrFunction sessionHook)
    : context(new pop_context())
    , settings(settings)
    , sessionLogger(createShortSessionInfo(*context))
    , networkSession(sessionHook)
{
    context->auth_cfg = settings.acfg;
}

Pop3Session::~Pop3Session()
{
    sessionLogger.log() << "closing with client " << context->sessionInfo.remote_address << ':'
                        << context->sessionInfo.remote_port << " (ssl=" << context->sslStatus
                        << "), list_requested=" << context->list_requested
                        << ", msize_requested=" << context->msize_requested
                        << ", muidl_requested=" << context->muidl_requested
                        << ", retr_messages=" << context->retr_messages
                        << ", deleted_messages=" << context->messages->deleted;

    yplatform::context_repository::instance().rem_context(context);
}

void Pop3Session::init(const yimap::SessionInfo& sessionInfo)
{
    L_(info) << context->uniq_id() << ":"
             << " new connection attempt:"
             << " remote_addr: " << sessionInfo.remote_address << ":"
             << std::to_string(sessionInfo.remote_port);

    context->sessionInfo = sessionInfo;
    context->sslStatus = settings.force_ssl ? "yes" : "no";
    context->tls_on = settings.force_ssl;
    yplatform::context_repository::instance().add_context(context);
    popProcessor = yplatform::find<Processor>("pop_processor");
}

void Pop3Session::greetings()
{
    networkSession()->clientStream()
        << "+OK POP Ya! na@" << SERVER_IDENTIFIER << " " << context->uniq_id() << "\r\n";
}

void Pop3Session::onTls(bool on, const string& sslChipher)
{
    context->tls_on = on;
    if (on)
    {
        context->sslStatus = "yes [" + sslChipher + "]";
    }
}

void Pop3Session::onReadError(const boost::system::error_code& e)
{
    if (e == boost::asio::error::eof)
    {
        sessionLogger.logEvent() << "handle_read: client closed connection, stopping reader";
        // } else if (filter().is_limit_exceed()) {
        //     sessionLogger.logError() << "handle_read: command size limit exceed, closing
        //     connection";
    }
    else
    {
        sessionLogger.logEvent() << "handle_read: " << e.message() << ", closing connection";
    }

    onShutdown();

    // networkSession()->postEvent(std::make_shared<ShutdownSession>());
}

void Pop3Session::onWriteError(const boost::system::error_code& e)
{
    sessionLogger.logEvent() << "handle_write error: '" << e.message() << "'";
}

void Pop3Session::processData(string&& chunk)
{
    buffer += chunk;
    std::size_t cmd_lenght = 0;

    request_t parsed_cmd(new command_t);
    if (!ypop::parse_command(buffer, *parsed_cmd, cmd_lenght))
    {
        networkSession()->postEvent(std::make_shared<StartReader>());
    }
    else
    {
        buffer.erase(0, cmd_lenght);
        popProcessor->process(parsed_cmd, context, networkSession());
    }
}

void Pop3Session::autologout()
{
    networkSession()->clientStream() << "-ERR Idle for too long\r\n";
}

void Pop3Session::onShutdown()
{
    if (context->mark_read_message_hook)
    {
        context->mark_read_message_hook();
        LOCK_POP_CONTEXT(*context);
        context->mark_read_message_hook = boost::function<void()>();
    }
}

} // namespace ypop
