#include <unistd.h>
#include <cstdio>
#include <cstdlib>
#include <cctype>
#include <cstring>
#include <algorithm>

#include <yplatform/repository.h>
#include <yplatform/find.h>
#include <yplatform/log.h>

#include <server/session.h>

namespace ypop {

Session::Session(yplatform::net::base_service* service, const ServerSettings& settings)
    : base_t(service, settings), popSession(settings, *this), readq_(this->create_buffer())
{
    filter().set_limit(settings.limits.max_packet_size);
}

yplatform::net::timer_ptr Session::createTimer(uint32_t milliseconds, yimap::VoidFunction hook)
{
    yplatform::time_traits::duration timeout = yplatform::time_traits::milliseconds(milliseconds);
    return create_timer(timeout, hook);
}

void Session::core_open()
{
    // context_->add_module_data("pop_server", this->stats());
    yimap::SessionInfo sessionInfo(
        protected_socket().raw().local_endpoint(), protected_socket().raw().remote_endpoint());
    popSession.init(sessionInfo);

    if (settings().force_ssl)
    {
        start_tls(boost::bind(&Session::start_session, shared_from_this(), _1));
    }
    else
    {
        start_session(boost::system::error_code());
    }
}

void Session::start_session(const boost::system::error_code& e)
{
    if (e) return;

    if (settings().force_ssl)
    {
        popSession.onTls(true, getSSLChiphers());
    }

    popSession.greetings();
    startWriter();
}

void Session::handle_read(
    const boost::system::error_code& e,
    std::size_t bytes,
    buffer_ptr /* buff */)
{
    if (e || filter().is_limit_exceed())
    {
        if (e && e != boost::asio::error::eof)
        {
            NET_SESSION_LOG(error) << "handle_read: " << e.message() << ", closing connection";
        }
        else if (filter().is_limit_exceed())
        {
            NET_SESSION_LOG(error) << "handle_read: command size limit exceed, closing connection";
        }
        else
        {
            NET_SESSION_LOG(info) << "handle_read: client closed connection, stopping reader";
        }
        shutdown(false);
        return;
    }
    processData(bytes);
}

void Session::processData(std::size_t bytes)
{
    if (popSession.isQuit())
    {
        shutdown(false);
        return;
    }
    if (!readq_->size())
    {
        startReader();
        return;
    }

    boost::asio::streambuf::const_buffers_type cbufs = readq_->data();
    std::string chunk(boost::asio::buffers_begin(cbufs), boost::asio::buffers_end(cbufs));
    readq_->consume(readq_->size());
    popSession.processData(std::forward<std::string>(chunk));
}

void Session::handle_write(const boost::system::error_code& e, std::size_t bytes)
{
    if (e)
    {
        NET_SESSION_LOG(error) << "handle_write error: '" << e.message() << "'";
        shutdown(false);
        return;
    }
    lock_t lock(mux_);
    writeq_.consume(bytes);
    if (startWriter()) return;

    if (tls_start_delayed)
    {
        start_tls(boost::bind(&Session::handle_tls_handshake, shared_from_this(), _1));
        NET_SESSION_LOG(debug) << "handle_write: tls started";
        tls_start_delayed = false;
    }
    else
    {
        lock.unlock();
        processData(0);
    }
}

void Session::handle_quit(const boost::system::error_code& e, std::size_t bytes)
{
    if (e)
    {
        NET_SESSION_LOG(error) << "handle_write error: '" << e.message() << "'";
        shutdown(false);
    }
}

string Session::getSSLChiphers()
{
    // typedef yplatform::net::stream::base_stream<protected_socket_t> base_stream_t;
    typedef yplatform::net::stream::ssl_stream<protected_socket_t> ssl_stream_t;

    auto& baseStream = stream();

    auto sslStream = dynamic_cast<ssl_stream_t*>(&baseStream);

    if (!sslStream)
    {
        return "error: no ssl stream";
    }

    SSL* ssl = sslStream->get_impl().native_handle();
    const SSL_CIPHER* cipher = SSL_get_current_cipher(ssl);
    int ssl_algbits = 0;
    int ssl_usebits = SSL_CIPHER_get_bits(cipher, &ssl_algbits);

    std::ostringstream chipherStream;
    chipherStream << SSL_get_version(ssl) << " with cipher " << SSL_CIPHER_get_name(cipher) << " ("
                  << ssl_usebits << "/" << ssl_algbits << " bits)";
    return chipherStream.str();
}

void Session::on_autologout()
{
    NET_SESSION_LOG(error) << "autologout timer exceed";
    lock_t lock(mux_);
    cancel_operations();
}

void Session::startReader()
{
    recv_data(
        boost::bind(&Session::handle_read, shared_from_this(), _1, _2, _3),
        boost::bind(&Session::on_autologout, shared_from_this()),
        readq_);
}

bool Session::startWriter()
{
    try
    {
        if (!writeq_.flush(this->settings().limits.max_output_chunk))
        {
            send_queue(
                writeq_.send_queue(),
                boost::bind(&Session::handle_write, shared_from_this(), _1, _2),
                boost::bind(&Session::on_autologout, this->shared_from_this()));
            return true;
        }
    }
    catch (...)
    {
        NET_SESSION_LOG(error) << "start_writer: ERROR during sending data to client";
    }
    return false;
}

yplatform::net::streamer_wrapper Session::client_stream()
{
    boost::shared_ptr<Session> s = shared_from_this();
    yplatform::net::streamer_wrapper wrp(new yplatform::net::streamer<Session>(s));
    return wrp;
}

void Session::send_client_stream(const yplatform::net::buffers::const_chunk_buffer& s)
{
    lock_t lock(mux_);
    if (protected_socket().raw().is_open()) writeq_.push(s);
}

// Working with TLS

void Session::handle_tls_handshake(boost::system::error_code const& e)
{
    if (e)
    {
        NET_SESSION_LOG(error) << "handle_tls_handshake error: " << e.message();
        return;
    }

    popSession.onTls(true, getSSLChiphers());

    NET_SESSION_LOG(debug) << "handle_tls_handshake: starting reader";
    startReader();
}

void Session::shutdown(bool graceful)
{
    {
        lock_t lock(mux_);
        if (session_finished)
        {
            return;
        }
        session_finished = true;
        do_shutdown(graceful);
    }

    popSession.onShutdown();
}

}
